	.TITLE	"FSK modem/TNC for HF RTTY and SITOR"
;
; See readme.txt file in this distribution for copyright notice, setup
; instructions and command summary.
;
; This program implements a FSK modem/TNC for HF asynchronous Baudot
; (ITA-2) and synchronous SITOR/AMTOR (CCIR 476 Mode B) signalling. It
; operates at speeds to 100 baud, either as a TNC with ASCII
; input/output via the onboard UART, or as a regenerator with clean AFSK
; output reconstructed from a possibly noisy input. The algorithms use
; Butterworth and matched filters, soft-decision detection, maximum a-
; priory (MAP) estimation and Viterbi decoding. In case of continuous
; asynchronous signalling, the modem can operate in a synchronous mode
; using a phase-lock loop (PLL) to flywheel through momentary carrier
; fades. The predetection design and many of the operator features were
; inspired by the Dovetron TU of long ago, but implemented in DSP,
; rather than analog form. The postdetection design, including MAP
; estimation, Viterbi and matched-filter decoding and synchronous mode
; was inspired by a LSI-11 program also of long ago. The algorithms
; implement a theoretically optimum receiver for FSK RTTY and SITOR
; signals in additive white Gaussian noise with nonfading and Rayleigh
; fading channels. If this thing can't pull RTTY and SITOR signals from
; the muck, nothing can.
;
; Include files
;
	.NOLIST
#INCLUDE "regs.inc"		; TMC320 registers
#INCLUDE "ports.inc"		; DSP93 ports
#INCLUDE "serial.inc"		; 16550 UART
	.LIST
;
; Program memory map
;
; External	1000-7fff	program
;
; Data memory map
;
; Block B2	0060-007f	interrupt variables and stack
; Block B0	0200-02ff	filter coefficients (mapped to prog mem)
; Block B1	0300-03ff	filter delay lines
; External	0400-7fff	program variables
;
; Auxiliary register usage
;
; AR0	utility compare and temporary register
; AR1	utility pointer
; AR2	utility counter/pointer
; AR3	utility pointer
; AR4	spare
; AR5	interrupt stack pointer
; AR6	AIO buffer pointer
; AR7	AIO buffer pointer
;
; Define macros
;
#DEFINE FAC 120*256/SFREQ	; Hz to phase increment
;
; Status bits (STATUS)
;
SYNSTA	.EQU	0		; sync mode
RSHSTA	.EQU	1		; receive Baudot figures shift
XSHSTA	.EQU	2		; transmit Baudot figures shift
RTSSTA	.EQU	3		; transmit request to send
CTSSTA	.EQU	4		; transmit clear to send
ERRSTA	.EQU	5		; error condition (buffer overrun)
;
; Mode bits (MODE)
;
SYNMOD 	.EQU	0		; sync mode enable
AUTMOD	.EQU	1		; autostart gate enable
NARMOD	.EQU	2		; narrow input filter
PORMOD	.EQU	3		; radio port 1/2 select
TORMOD	.EQU	4		; SITOR Mode B enable
ECHMOD	.EQU	5		; digital echo on transmit
FDXMOD	.EQU	6		; full-duplex
REVMOD	.EQU	7		; reverse shift
RUSMOD	.EQU	8		; receive unshift-on-space
XUSMOD	.EQU	9		; transmit unshift-on-space
ERSMOD	.EQU	10		; suppress erasures
MRKMOD	.EQU	11		; mark channel disable
SPCMOD	.EQU	12		; space channel disable
;
; AIO Configuration constants: HPF in, sync, 3-V half scale, sin x/x.
; RA is chosen to get the SCF as close to 288 kHz as possible; RB is
; chosen to get the sample frequency as close to 8.0 kHz as possible.
; This gives a 3-dB bandwidth of about 100-3700 Hz at the AIO input.
;
CFGA	.EQU	CMDA_VAL(17)	; AIO load RA=TA=17 for SCF 294.1 kHz
CFGB	.EQU	CMDB_VAL(37)	; AIO load RB=TB=37 for 7.949 kHz
AIO_GAIN .EQU	AIO_GAIND60|AIO_GAIND70 ; AIO gain
CFGC	.EQU	AIO_CONFIG|AIO_INSADHPF|AIO_SYNC|AIO_GAIN|AIO_INSSINX
;
; Tone frequency definitions 
;
DEGREE	.EQU	360/3		; sin table entries (3-degree steps)
PI2	.EQU	DEGREE<<8	; radix for output phase wrap
SFREQ	.EQU	8000		; sample frequency (Hz)
INCR	.EQU	((17*37)<<(16-8))/5 ; sample interval (q6)
TUNEX	.EQU	(10*120*256)/SFREQ ; 10 Hz
SITBIT	.EQU	80		; samples per SITOR encoder bit
;
; Sizes of arrays
;
BINSIZ	.EQU	8		; samples per baud interval
CHRSIZ	.EQU	8		; baud intervals per character + 1
SRVSIZ	.EQU	CHRSIZ*BINSIZ	; size of survivor table
CARSIZ	.EQU	SRVSIZ*3	; size of carrier/noise delay line
MFRSIZ	.EQU	3000		; max size of matched filter delay line
ZONE	.EQU	300		; size of red and yellow buffer zones
OUTSIZ	.EQU	1000		; size of character circular buffers
FIRSIZ	.EQU	71		; size of FIR bpf delay line
TORSIZ	.EQU	7*6		; size of SITOR character shift register
AUTSIZ	.EQU	15		; size of autostart shift register (chars)
IIRSIZ	.EQU	3*4		; size of IIR lpf delay lines (4)
STRSIZ	.EQU	16		; size of filter structure
BUFSIZ	.EQU	50		; size of AIO buffer
;
; Miscellaneous definitions
;
LIMIT	.EQU	10000		; limiter threshold (LED 1)
CLIMIT	.EQU	8000		; carrier preset threshold
ELIMIT	.EQU	23000		; erasure preset threshold
ALIMIT	.EQU	25000		; autostart preset threshold
FRMSIZ	.EQU	SRVSIZ-BINSIZ	; sync counter radix (samples / char)
CCONST	.EQU	16-4		; scaled carrier/noise time constant
TCONST	.EQU	16-4		; RTTY acquistion time constant
SCONST	.EQU	3		; RTTY sync time constant (right shift)
MCONST	.EQU	7		; SITOR pll time constant (right shift)
P1_MSK	.EQU	PTT_R1+FREQ_UP_R1+FREQ_DN_R1 ; RADIO_CTRL port 1 mask
P2_MSK	.EQU	PTT_R2+FREQ_UP_R2+FREQ_DN_R2 ; RADIO_CTRL port 2 mask
TUNEW	.EQU	10*8		; tune pulse width (ms * 8))
AHOLD	.EQU	200		; RTTY autostart hold interval (chars)
RHOLD	.EQU	10		; RTS hold interval (chars)
PIO2	.EQU	51472>>1	; pi / 2 in q14 format
RQ	.EQU	1100110b	; CCIR 476 signal RQ
ALPHA	.EQU	0001111b	; CCIR 476 signal alpha
;
; Nonprinting and special character codes
;
SP	.EQU	04h		; ITA-2 space
FIGS	.EQU	1Bh		; ITA-2 FIGS shift
LTRS	.EQU	1Fh		; ITA-2 LTRS shift
NUL	.EQU	00h		; ITA-2/ASCII null
EOT	.EQU	04h		; ASCII EOT (down PTT immediately)
ENQ	.EQU	05h		; ASCII enquiry/answerback request
ACK	.EQU	06h		; ASCII ACK (down PTT when buffer empty)
BEL	.EQU	07h		; ASCII bell
LF	.EQU	0Ah		; ASCII	line feed
CR	.EQU	0Dh		; ASCII carriage return
SYN	.EQU	16h		; ASCII SYN (Baudot LTRS/diddle)
LCTST	.EQU	5		; ITA-2 LTRS test
LC	.EQU	1<<LCTST	; ITA-2 LTRS bit
UCTST	.EQU	6		; ITA-2 LTRS test
UC	.EQU	1<<UCTST	; ITA-2 FIGS bit

;
; Interesting block addresses and memory page pointers
;
B_PAGE	.EQU	0h		; base page pointer
U_PAGE	.EQU	8h		; user page pointer
U_ADDR	.EQU	U_PAGE<<7	; user page address
B0P	.EQU	0FF00h		; block B0 program address
B0D	.EQU	200h		; block B0 data address
B1D	.EQU	300h		; block B1 data address

;
; Following are variables accessed by either absolute or indirect
; addressing modes.
;
; Program variables in internal memory (block B0) initialized by
; program. These include FIR and IIR filter coefficients, as well as the
; SITOR bit vector delay line. The first address in each set of two is
; valid following a CNFD instruction, which maps block B0 to data
; memory; the second address is valid following a CNFP instructuion,
; which maps block B0 to program memory.
;
BPFCOD	.EQU	B0D		; FIR bandpass filter coefficients
BPFCOP	.EQU	BPFCOD+B0P-B0D
LPFCOD	.EQU	BPFCOD+FIRSIZ	; IIR lowpass filter coefficients
LPFCOP	.EQU	LPFCOD+B0P-B0D
TORDLD	.EQU	LPFCOD+IIRSIZ	; SITOR bit vector delay line
TORDLP	.EQU	TORDLD+B0P-B0D
B0DEND	.EQU	TORDLD+(6*7)	; end of block B0
;
; Program variables in internal memory (block B1) cleared at reset.
; These consist of delay lines used by the MACD and DMOV instructions,
; which require the data to be in internal memory.
;
DELBPF	.EQU	B1D		; FIR bandpass filter delay line
DELLPF	.EQU	DELBPF+FIRSIZ+1	; IIR lowpass filter delay line
;
; The survivor table contains a shift-register survivor for eight
; samples of each bit and eight bits of the character. It is used to
; construct the maximum and minimum amplitudes, as well as the slice
; level, carrier and noise estimates.
;
SRVBGN	.EQU	DELLPF+IIRSIZ	; beginning of survivor table
SRVEND	.EQU	SRVBGN+SRVSIZ	; end of survivor table
;
; Program variables in external memory (U_PAGE) cleared at reset. We
; protect one location at the beginning for I/O status.
;
USRBGN	.EQU	U_ADDR+1	; beginning of clear region
;
; Filter structures are used by the FIR, IIR and MF filters. For the IIR
; and MF filters, the mark and space filter center frequencies are
; determined by the phase increment. For these filters, the mark filter
; is fixed and the space filter is tunable. The FIR filters are not
; tunable.
;
; Filter structure format
; 0	filter routine entry point
; 1	phase increment
; 2	current phase
; 3-4	I channel output
; 5-6	Q channel output
; 7-8	sum of squares
; 9-10	detector output
; 11-15 not used
;
MSTRUCT	.EQU	U_ADDR+100h	; mark filter structure
SSTRUCT	.EQU	MSTRUCT+STRSIZ	; space filter structure
;
; The AIO buffer is a circular structure indexed by AR6 and AR7. AR6 is
; used only by interrupt routines. The receive interrupt routine stores
; the sample at AR6 and then increments AR6. The transmit interrupt
; routine loads the sample at AR6, which is the oldest in the circular
; buffer. The main program loops until AR7 is not equal to AR6, then
; loads the sample found at AR7, which is the oldest sample not yet
; processed. The program loads the output at AR7 and then increments
; AR7. Since input and output interrupts are synchronous, no data are
; lost or duplicated.
;
INTBUF	.EQU	SSTRUCT+STRSIZ	; AIO buffer
;
; Circular buffers and buffer structures are used for character I/O, one
; set each for input (modem and echo to UART) and output (UART to
; modem).
;
; Buffer structure format
; 0	address of first word beyond buffer
; 1	address of first word of buffer
; 2	put pointer
; 4	get pointer
;
INPBCB	.EQU	INTBUF+BUFSIZ	; input buffer structure
OUTBCB	.EQU	INPBCB+4	; output buffer structure
INPBUF	.EQU	OUTBCB+4	; transmit circular buffer
OUTBUF	.EQU	INPBUF+OUTSIZ	; receive circular buffer
;
; This is a boxcar integrator used as a predetection and postdetection
; matched filter. It can grow up to 3000 words, depending on baud
; interval. In the predetection case, it contains four samples for each
; cycle; in the baseband case, it contains two samples interleaved with
; two dummy words for a total of four.
;
DELMFR	.EQU	OUTBUF+OUTSIZ	; matched filter delay line
;
; The following delay lines are used to determine the carrier and
; autostart distances. The first (DELCAR) is a set of boxcar integrators
; for the carrier and noise estimates. The second (DELNOI) is used as a
; signal and noise classifier to pry loose legitimate carrier and noise
; samples from rotten modulation products. The third (AUTDEL) is a
; boxcar integrator for the autostart function.
;
DELCAR	.EQU	DELMFR+MFRSIZ	; carrier/noise delay line
DELNOI	.EQU	DELCAR+(2*CARSIZ) ; carrier/noise classifier delay line
AUTDEL	.EQU	DELNOI+BINSIZ	; autostart delay line
;
; These arrays are used by the SITOR decoder to integrate the distance
; for each of 14 phase positions and five repetitions of the CCIR 476
; sync sequence RQ-ALPHA.
;
TORSTR	.EQU	AUTDEL+(2*AUTSIZ) ; phase delay lines
TORACC	.EQU	TORSTR+(14*2*5)	; phase/error accumulators
;
; Table of FIR coefficients for the SITOR matched-filter receiver. We
; run through these to find the most likely vector with four bits one
; and three bits zero. There are 35 of these suckers, 32 of which have
; Baudot equivalents and three of which are special characters.
;
TORTAB	.EQU	TORACC+(14*2)	; SITOR coefficients
USREND	.EQU	TORTAB+(35*7)	; end of clear region

;
; Following are variables accessed by direct addressing mode. These are
; offsets relative the the data page pointer.
;
: Program variable offsets in block B2 (B_PAGE) initialized at startup.
;
TNC_BUF	.EQU	60h		; TNC_OUTPUT port status
PDC_BUF	.EQU	61h		; RADIO_CTRL port status
;
; Program variable offsets in block B2 (B_PAGE) uninitialized.
;
UARTI	.EQU	62h		; UART input buffer
UARTO	.EQU	63h		; UART output buffer
UARTC	.EQU	64h		; UART modem control buffer
XTMP	.EQU	65h		; temp
SPY	.EQU	70h		; spy temp (debug)
STACK	.EQU	80h		; interrupt stack (grows down)
;
; Program variable offsets in external memory (U_PAGE) initialized at
; startup.
;
PIO_BUF	.EQU	00h		; RADIO_GAIN port status
;
; Program variable offsets in external memory (U_PAGE) cleared at reset.
;
STATUS	.EQU	01h		; status bits
MODE	.EQU	02h		; mode bits
;
; RF analog signal variables and pointers
;
INPSIG	.EQU	03h		; AIO input buffer
FLTSIG	.EQU	04h		; filtered/limited input signal
MARK	.EQU	05h		; mark channel output
SPACE	.EQU	06h		; space channel output
MFPTR	.EQU	07h		; matched filter pointer
MFSIZE	.EQU	08h		; matched filter size
MFGAIN	.EQU	09h		; matched filter gain
OUTSW	.EQU	0Ah		; display pointer
OMARK	.EQU	0Bh		; AFSK mark phase increment
OSPACE	.EQU	0Ch		; AFSK space phase increment
OPHASE	.EQU	0Dh		; AFSK output phase
;
; Baseband analog signal variables and pointers
;
ACCUM	.EQU	10h		; scope detector output
SSYNC	.EQU	11h		; scope sync
POSMAX	.EQU	12h		; max positive signal
NEGMAX	.EQU	13h		; max negative signal
SLICE	.EQU	14h		; comparator slice level
CARITG	.EQU	15h		; (2) carrier integrator
NOSITG	.EQU	17h		; (2) noise estimator
CARPTR	.EQU	19h		; carrier/noise pointer
NOIPTR	.EQU	1Ah		; carrier/noise classifier pointer
;
; Distances, thresholds and scope monitors. The gates are used also as
; scope monitors to provide visual reference when adjusting thresholds.
;
CTHRESH	.EQU	20h		; carrier gate threshold (c{+-} command)
CSCOPE	.EQU	21h		; carrier gate
ERSDST	.EQU	22h		; erasure distance
ETHRESH	.EQU	23h		; erasure gate threshold (v{+-} command)
ESCOPE	.EQU	24h		; erasure gate
AUTSIG	.EQU	25h		; autostart signal
ATHRESH	.EQU	26h		; autostart gate thresh (a{+-} command)
ASCOPE	.EQU	27h		; autostart gate
LEDREF	.EQU	28h		; mark/space LED reference
;
; Tuning and gain controls
;
COUNT	.EQU	40h		; (2) baud interval counter VCO
SYNCTR	.EQU	42h		; (2) RTTY sync counter VCO
RANGE	.EQU	44h		; baud interval counter radix
XPHASE	.EQU	45h		; phase error signal
GAIN	.EQU	46h		; filter gain (a{+-} command)
LGAIN	.EQU	47h		; limiter gain (shift) (l{0-5} command)
AGAIN	.EQU	48h		; analog gain (a{0-7} command)
;
; Peekaboo for debug/monitor (undocumented d{-_} command)
;
PEEK1	.EQU	49h		; debug peekaboo 1
PEEK2	.EQU	4Ah		; debug peekaboo 2
;
; Waveform decoder variables
;
XSMCTR	.EQU	50h		; (2) sample counter
SQRACC	.EQU	52h		; (2) sum of squares
TORPTR	.EQU	54h		; SITOR sync delay line pointer
TORACP	.EQU	55h		; SITOR sync accumulator pointer
SYNDEL	.EQU	56h		; RTTY frame interval counter
SMPCTR	.EQU	57h		; sample counter
BITCTR	.EQU	58h		; bit counter
SRVCTR	.EQU	59h		; strobe counter
AUTCTR	.EQU	5Ah		; autostart counter
CHRCTR	.EQU	5Bh		; character counter
AUTPTR	.EQU	5Ch		; autostart delay line pointer
AUTITG	.EQU	5Dh		; autostart signal integrator
AUNITG	.EQU	5Eh		; autostart noise integrator
;
; Waveform encoder variables
;
KEY	.EQU	60h		; (3) encoder shift register
FECCHR	.EQU	63h		; character buffer
XBTCTR	.EQU	64h		; bit counter
XCHCTR	.EQU	66h		; character counter
RTSCTR	.EQU	67h		; RTS counter
;
; Radio control and panel display variables
;
TUNE	.EQU	70h		; radio port tune indicator
TUNCTR	.EQU	71h		; tune pulse counter
FLASH	.EQU	72h		; utility counter for flashers
;
; Command interpreter variables
;
UTMP1	.EQU	73h		; user temp 1
UTMP2	.EQU	74h		; user temp 2
UCHAR	.EQU	75h		; character temp
UPTR	.EQU	76h		; pointer temp
UADR	.EQU	77h		; address temp
UCMD	.EQU	78h		; command table pointer
UMOD	.EQU	79h		; saved command table pointer
;
; Wacky temporaries
;
TMP	.EQU	7Ah		; global temporary
STS	.EQU	7Bh		; global temporary
STMP1	.EQU	7Ch		; local temporary
STMP2	.EQU	7Dh		; local temporary

;
;  Program transfer vector
;
	.ORG	1000h		; monitor origin
	B       TINT		; timer interrupt
	B       RINT		; AIO receive interrupt
XIN10	B       XINT		; AIO transmit interrupt
	B       IINT		; TRAP instruction interrupt
	B       GO		; program starting address

;
; AIO receive interrupt. This is designed for minimal intrusion on
; variables used by interruptable code. AIO rx interrupt is enabled when
; tx initialization is complete.
;
; Note: This is the only place where AR6 is incremented. Immediately
; after increment, AR6 is checked for overflow and wrapped.
;
; Variables
; AR6		AIO buffer pointer
; DRR		AIO input buffer
;
RINT	LARP	AR5		; save context
	MAR	*-
	SST1	*-
	SST	*-
	SAR	AR0,*,AR6	; AR6 is AIO buffer pointer
	LDPK	B_PAGE
	LAR	AR0,DRR		; read AIO input buffer
	SAR	AR0,*+
	LRLK	AR0,INTBUF+BUFSIZ ; is pointer at buffer end
	CMPR	
	BBZ	RIN10		; branch if no
	LRLK	AR6,INTBUF	; yes. reset pointer
RIN10	LARP	AR5
	CALL	DUMP		; dump status
	LAR	AR0,*+		; restore context
	LST	*+
	LST1	*+
	EINT			; enable interrupts
	RET

;
; AIO transmit interrupt used in normal processing
;
; Variables
; AR6		AIO buffer pointer
; DXR		AIO output buffer
;
XINT	LARP	AR5		; save context
	MAR	*-
	SST1	*-
	SST	*-
	SAR	AR0,*,AR6	; AR6 is AIO buffer pointer
	LDPK	B_PAGE
	LAR	AR0,*,AR5	; write AIO output buffer
	SAR	AR0,DXR
	CALL	DUMP		; dump status
	LAR	AR0,*+		; restore context
	LST	*+
	LST1	*+
	EINT			; enable interrupts
	RET

;
; AIO transmit interrupt routine used only during AIO configuration.
; Note that the ARP and designated AR point to configuration data.
;
; Variables
; DXR		AIO output buffer
;
XINT2	LAC	*+		; this has to be quick and ugly
	SACL	DXR
	EINT
	RET

;
; Trap interrupt (for debug). TRAP instructions can be sprinkled in the
; code to watch critical sections and leave tracks useful for debugging.
;
IINT	DINT			; avoid interrupts here
	LARP	AR5		; save context
	MAR	*-
	SST1	*-
	SST	*-
	SAR	AR0,*
	LDPK	B_PAGE
	CALL	DUMP		; dump status
	LAR	AR0,*+		; restore context
	LST	*+
	LST1	*+
	EINT			; enable interrupts
	RET

;
; Routine to dump trace data for debug. This is done on every I/O and
; TRAP interrupt. At this point, SS1, SS0 and AR0 are already on the
; stack (AR5) and the data page pointer set to B_PAGE (0). The remaining
; registers and top two words on the internal CPU stack are saved.
;
DUMP	SAR	AR5,XTMP	; save stack pointer
	MAR	*-
	SAR	AR1,*-		; save aux registers AR1-AR7
	SAR	AR2,*-
	SAR	AR3,*-
	SAR	AR4,*-
	SAR	AR5,*-
	SAR	AR6,*-
	SAR	AR7,*-
	POPD	*-		; save calling address
	POPD	*		; save interrupting address
	PSHD	*+
	PSHD	*-
	LAR	AR5,XTMP	; restore stack pointer
	RET

;
; Timer interrupt (not used)
;
TINT	EINT			; enable interrupts
	RET

;
; Initialize processor configuration. These set the options as does the
; CPU upon first coming up; they are here in case the monitor screws up.
;
GO	CNFP			; configure block B0 as program
	SSXM			; set sign extension mode
	SOVM			; set overflow mode
	SPM	1		; set P register output shift for q15
	FORT	0		; set AIO 16-bit mode
	SFSM			; set FSM (1) frame sync mode
	RTXM			; set TXM (0) transmit mode
	LDPK	U_PAGE
	LALK	AIO_RESET+GAIN4+REC_IN_1 ; reset AIO
	SACL	PIO_BUF
	OUT	PIO_BUF,RADIO_GAIN
	CALL    WAIT4
	LALK    LED_202+GAIN4+REC_IN_1 ; disable AIO; LED 2 on (debug)
	SACL    PIO_BUF
	OUT     PIO_BUF,RADIO_GAIN 
	CALL    WAIT4
	LALK    AIO_ENABLE+GAIN4+REC_IN_1 ;enable AIO
	SACL    PIO_BUF        
	OUT     PIO_BUF,RADIO_GAIN
	CALL    WAIT4
	LDPK	B_PAGE
	LALK	UART_MCR_RTS	; raise RTS (CTS at connector)
	SACL	UARTC
	RPTK	7-1		; flush stack
	POP
	LRLK	AR5,STACK	; initialize stack pointer
	LRLK	AR6,INTBUF	; initialize AIO buffer pointers
	LRLK	AR7,INTBUF
;
; Set up for AIO initialization. Keep this a deep dark secret and don't
; tell anybody the transmit interrupt routines are switched on-the-wing
; for configuration. This is because the AIO rascal has unreal latency
; requirements only during initialization. There seems to be no way for
; an interrupt routine to satisfy these without appearing non-
; transparent for general purpose processing.
;
	DINT			; turn out the lights
	LALK	XINT2
	SACL	XTMP
	LALK	XIN10+1		; clobber transmit vector
	TBLW	XTMP
	LARP	AR7		; copy AIO initialization vector
	RPTK	7-1
	BLKP	INITAB,*+
	LACK	20h		; enable AIO tx interrupt
	SACL	IMR
	ZAC			; flush AIO tx buffer
	SACL	DXR
	EINT			; turn on the lights
	LARP	AR6		; load initialization word
INI10	IDLE			; wait for interrupt and AC
	BNZ	INI10		; branch if more
	DINT			; done. darkness again
	LALK	XINT		; unclobber transmit vector
	SACL	XTMP
	LALK	XIN10+1
	TBLW	XTMP
	LACK	30h		; enable AIO rx and tx interrupts
	SACL	IMR
	LDPK	U_PAGE
	CALL	RESET		; initialize program variables
	EINT			; enable interrupts
;
; Loop looking for a AIO input sample. When found, the RF input
; processing routine is called to filter and limit the input signal and
; develop the mark and space baseband signals. The RF output processing
; routine is also called to generate the AIO output signal. The input
; samples are decimated at 8 samples per bit interval. At each sample
; the baseband input routine is called to generate the input FSK keying
; waveform and decode the ASCII character. Then, the baseband output
; routine is called to encode the ASCII character and generate the
; output keying waveform.
;
; Variables
; RANGE		baud interval counter radix
; COUNT		baud interval counter VCO
; TUNE		radio port tune indicator
; TUNCTR	tune pulse counter
;
; Note: This is the only place where AR6 is compared to AR7. The CMPR
; insruction represents the atomic interaction in order to determine
; whether data are in the buffer or not; thus, there are no races. Even
; if there were, unread data would get caught on the next cycle through
; the loop.
;
LOOP	LARP	AR7		; AR7 is AIO buffer pointer
	SAR	AR6,TMP		; are there data in the buffer
	LAR	AR0,TMP
	CMPR	
	BBZ	LOP10		; branch if yes
	LALK	LED_202		; no. LED 2 on
	CALL    LEDON
	CALL	LIGHTS		; blink the control panel
	CALL	GETCH		; UART input processing
	CALL	PUTCH		; UART output processing
	LALK	LED_202		; LED 2 off
	CALL    LEDOFF
	LAC	TUNCTR		; is tune pulse counter running
	BZ	LOOP		; branch if no
	SUBK	1		; yes. decrement by 125 us
	SACL	TUNCTR		; did counter underflow
	BGZ	LOOP		; branch if no
	ZAC			; yes. kill the tune bits
	SACL	TUNE
	B       LOOP
;
; A new input sample is in the AIO input buffer. The processing routines
; are called according to the following decimation schedule:
;
; AIO sample rate. This occurs at the (adjusted) sample interval of
; SFREQ (8000 Hz). The baseband mark and space signals are developed
; here along with the AFSK output signals. The sample interval is
; determined using a two-word counter in microseconds and fraction. The
; increment is in microseconds shifted left by eight bits, which results
; in a frequency resolution of about 0.4 ppm at 100 baud.
;
; Baseband sample rate. This occurs at eight times the nominal baud
; rate, but is variable over a small range for SITOR symbol tracking.
; The maximum and minimum differential signals, slice level and
; mark/space memory signals are developed here. The RTTY and SITOR input
; processing is done here using matched-filter techniques in which the
; calculated distance is used as a quality indicator.
;
; Bit rate. This occurs at the programmed baud rate, which is as
; close to the specified baud rate as possible. The RTTY and SITOR
; output processing is done here and the carrier and noise estimates are
; developed.
;
; Character rate. This occurs at the nominal character rate, which is
; 7.5 bit intervals for RTTY and 14 bit intervals for SITOR.
;
LOP10	CALL	RFINP		; RF input processing
	CALL	RFOUT		; RF output processing
	BIT	MODE,TORMOD	; is this SITOR
	BBZ	LOP13		; branch if no
	ZALH	XSMCTR		; yes. is it time for the next bit
	ADDS	XSMCTR+1
	SBLK	INCR,8-3	; (note alignment - 0-128 us plus sign)
	SACH	XSMCTR
	SACL	XSMCTR+1
	BGEZ	LOP13		; branch if no
	ADDH	RANGE		; yes. advance one sample interval
	SACH	XSMCTR
	CALL	SITOUT		; SITOR encoder processing
LOP13	ZALH	COUNT		; is it time for the next sample
	ADDS	COUNT+1
	SBLK	INCR,8		; (note alignment - 0-128 us plus sign)
	SACH	COUNT
	SACL	COUNT+1
	BGEZ	LOOP		; branch if no
	ADDH	RANGE		; yes. advance one sample interval
	SACH	COUNT
	CALL	DSPINP		; baseband input processing
	CALL	BITS
	BIT	MODE,TORMOD	; is alternate decoder selected
	BBNZ	LOP12		; branch if yes
	CALL	DSPOUT		; no. RTTY encoder processing
	CALL	SSTOP		; RTTY decoder processing
	B	LOOP
;
LOP12	CALL	SITOR		; SITOR decoder processing
	B	LOOP

;
; RF input processing
;
; This routine processes the AIO input signal to produce the mark and
; space signals used by the baseband input routine.
;
; Variables
; AR7		AIO buffer pointer
; INPSIG	AIO input buffer
; FLTSIG	filtered/limited input signal
; MARK		mark channel output
; SPACE		space channel output
; DELBPF	FIR bandpass filter delay line
; DELLPF	IIR lowpass filter delay line
; MSTRUCT	mark filter structure
; SSTRUCT	space filter structure
;
; Clip signals with energy above an arbitrary threshold (LIMIT). This
; helps to reduce errors due to noise spikes and static crashes, etc.
; The threshold is chosen rather arbitrary at 10dB below overload. LED 1
; is lit when in limiting condition.
;
RFINP	LARP	AR7		; AR7 is AIO buffer pointer
	LAC	*,2,AR1		; load sample in q15 format
	SACL	INPSIG
	ABS			; is signal above threshold
	SBLK	LIMIT
	BLZ	INP10		; branch if no
	LALK	LED_201		; yes. LED 1 on
	CALL	LEDON
	B	INP11
;
INP10	LALK	LED_201		; signal below threshold. LED 1 off
	CALL	LEDOFF
INP11	LAC	INPSIG		; clip signal at max
	SBLK	LIMIT
	BLZ	INP12
	ZAC
INP12	ADLK	2*LIMIT		; clip signal at min
	BGZ	INP13
	ZAC
INP13	SBLK	LIMIT
	SACL	FLTSIG
;
; Filter the clipped signal using a FIR filter. The filter bandwidth is
; determined by the coefficients loaded in block B0P. A narrow (about
; 400 Hz) filter is used for 170-Hz shift and wide (about 1000 Hz)
; filter used for the wider shifts. These filters are used primarily to
; reduce the effects of aliasing in the following limiter. The l{0-5}
; command is used to select the limiter gain.
;
	LRLK	AR1,DELBPF
	LAC	FLTSIG
	SACL	*
	ADRK	FIRSIZ-1	; point to end of delay line
	MPYK	0		; initialize sum and product registers
	ZAC
	RPTK	FIRSIZ-1	; multiply and accumulate
	MACD	BPFCOP,*-
	APAC			; include last product
	SACH	FLTSIG
;
; Limit signal at selected gain 0-30dB. Under good conditions, up to 30-
; dB limiting can be used with IIR filters for ease of tuning. When
; things get tough, use less limiting and matched filters.
; 
	LT	LGAIN		; get limiter gain
	LACT	FLTSIG		; limit signal at max
	SBLK	LIMIT
	BLZ	INP20
	ZAC
INP20	ADLK	2*LIMIT		; limit signal at min
	BGZ	INP21
	ZAC
INP21	SBLK	LIMIT
	SACL	FLTSIG
;
; The demodulator center frequency and mark-space shift is selected with
; the m{1-8} command. The center frequency is 1700 Hz for wide shifts
; (850, 600, and 400 Hz) and 2210 Hz for narrow shift (170 Hz). For the
; synchronous filters, the receiver is first tuned to the mark
; frequency, which is fixed at 1275, 1400, 1500 and 2125 Hz for 850,
; 600, 400 and 170-Hz shifts, respectively. At these shifts the nominal
; space frequency is 2125, 2000. 1900 and 2295 Hz, respectively. The t+
; and t- commands can be used to fine-tune the space frequency. The
; asynchronous filter is not tunable. These particular choices are
; compatible with the AEA PK232 TNC. Note that the mark and space
; channels can be switched or either one can be disabled.
;
	LRLK	AR2,DELLPF	; set shared IIR delay line pointer
	LRLK	AR1,MSTRUCT	; process mark channel
	LAC	*+
	CALA
	BIT	MODE,MRKMOD	; is channel disabled
	BBZ	INP33		; branch if no
	ZAC			; yes. kill it
INP33	SACL	MARK		; save mark
	LRLK	AR1,SSTRUCT	; process space channel
	LAC	*+
	CALA
	BIT	MODE,SPCMOD	; is space channel disabled
	BBZ	INP34		; branch if no
	ZAC			; yes. kill it
INP34	SACL	SPACE		; save space
	LAC	MFPTR		; adjust MF filter pointer
	SBLK	DELMFR
	SUB	MFSIZE
	BLZ	INP32
	LALK	DELMFR
	SACL	MFPTR
INP32	RET

;
; RF output processing
;
; This routine synthesizes the AIO output signal from the FSK waveform
; produced by the baseband output routine.
;
; Variables
; AR7		AIO buffer pointer
; KEY		encoder shift register
; OUTSW		display pointer
; OMARK		AFSK mark phase increment
; OSPACE	AFSK space phase increment
; OPHASE	AFSK output phase
;
; The AFSK mark and space frequencies are determined according to mode
; and tables. In receive mode, provision is made to monitor the AFSK
; output, as well as various signals and state variables selected by the
; d{0-9} command. In transmit mode, the signal generated is always the
; AFSK output. Note that the AFSK output sinewave is attenuated about
; 10dB, in order to match the nominal receiver signal level.
;
; Note: This is the only place where AR7 is incremented. Immediately
; after increment, AR7 is checked for overflow and wrapped.
;
RFOUT	LARP	AR1		; AR1 means business
	BIT	KEY,0		; is lsb bit one
	BBZ	OUT11		; branch if no
	LAC	OMARK		; yes. set mark interval
	B	OUT12
;
OUT11	LAC	OSPACE		; lsb zero. set space interval
OUT12	ADD	OPHASE		; decrement phase
	BGEZ	OUT10		; branch if no underflow
	ADLK	PI2		; underflow. add 2 * pi
OUT10	SACL	OPHASE
	LAC	OPHASE,8	; make table pointer
	SACH	UTMP1
	LALK	SINTAB
	ADD	UTMP1
	TBLR	STS		; convert to sin amplitude
	LALK	LIMIT		; scale for convenient A/B compare
	SACL	TMP
	LT	TMP
	MPY	STS
	PAC
	SACH	STS
	LAC	STS
	BIT	STATUS,CTSSTA	; is this transmit mode
	BBNZ	OUT14		; branch if yes
	LAR	AR1,OUTSW	; no. fetch source signal in q15 format
	LAC	*
OUT14	ANDK    0FFFCh
	LARP	AR7		; AR7 is AIO buffer pointer
	SACL	*+		; store output sample in q15 format
	LRLK	AR0,INTBUF+BUFSIZ ; is pointer at buffer end
	CMPR	
	BBZ	OUT13		; branch if no
	LRLK	AR7,INTBUF	; yes. reset pointer
OUT13	RET

;
; RTTY/SITOR baseband input processing
;
; This routine processes the baseband mark and space signals produced by
; the RF input routine and generates useful signals used by the
; character decoder routines. It is called once each sample interval,
; which occurs at eight times the baud rate. The signals produced
; include the carrier and noise estimates, maximum and minimum envelope
; estimates, and comparator slice level and autostart signal.
;
; Variables
; MARK		mark channel output
; SPACE		space channel output
; POSMAX	max positive signal
; NEGMAX	max negative signal
; SLICE		comparator slice level
; SRVBGN	survivor table
;
; Determine the mark-space differential signal and save it in the
; survivor table. The selectable shift invert is done here.
;
DSPINP	LARP	AR1		; AR1 means business
	LAC	MARK		; compute differential signal
	SUB	SPACE
	BIT	MODE,REVMOD	; is shift reversed
	BBZ	DSI11		; branch if no
	NEG			; yes. invert signal
DSI11	LRLK	AR1,SRVBGN+SRVSIZ-2 ; point to end of survivor table
	RPTK	SRVSIZ-2	; shift old samples
	DMOV	*-
	MAR	*+
	SACL	*		; insert new sample first
	CALL	MODEL12		; generate scope sync signal
;
; Compute the maximum and minimum of all survivors. The loop is unrolled
; twice in order to reduce overhead; however, the frequent branches
; effectively stall the pipeline.
;
	LRLK	AR1,SRVBGN	; point to beginning of survivor table
	LRLK	AR0,SRVSIZ+SRVBGN ; point to end of survivor table
	LALK	8000h		; initialize for max search
	SACL	POSMAX
	LALK	03777h		; initialize for min search
	SACL	NEGMAX
DSI21	LAC	*		; is sample > max
	SUB	POSMAX
	BLEZ	DSI22		; branch if no
	ADD	POSMAX		; yes. update max
	SACL	POSMAX
DSI22	LAC	*+		; is sample < min
	SUB	NEGMAX
	BGEZ	DSI23		; branch if no
	ADD	NEGMAX		; yes. update min
	SACL	NEGMAX
DSI23	LAC	*		; is sample > max
	SUB	POSMAX
	BLEZ	DSI24		; branch if no
	ADD	POSMAX		; yes. update max
	SACL	POSMAX
DSI24	LAC	*+		; is sample < min
	SUB	NEGMAX
	BGEZ	DSI25		; branch if no
	ADD	NEGMAX		; yes. update min
	SACL	NEGMAX
DSI25	CMPR			; is search complete
	BBZ	DSI21		; branch if no
	LAC	POSMAX		; compute slice level
	ADD	NEGMAX
	SFR
	SACL	SLICE
	RET
;
; This routine teases out good signal and noise estimates and controls
; the signal gate. It is called once each sample interval, which occurs
; at eight times the baud rate. 
;
; Variables
; MARK		mark channel output
; SPACE		space channel output
; CARITG	carrier integrator
; NOSITG	noise integrator
; LEDREF	mark/space LED reference
; CTHRESH	carrier threshold
; DELCAR	carrier/noise integrator
; CARPTR	carrier/noise pointer
; DELNOI	carrier/noise classifier delay line
; NOIPTR	carrier/noise classifier pointer
;
; The energy (signal square) of each sample of the mark and space
; channels is computed. If the mark energy is at least twice the space
; energy, the mark signal is a carrier sample and the space signal is a
; noise sample. If the space energy is at least twice the mark energy,
; the space signal is a carrier sample and the mark signal is a noise
; sample. If neither is the case, both the mark and space signals are
; noise samples.
;
BITS	LARP	AR1		; go where the action is
	LAC	MARK		; does mark dominate
	SUB	SPACE,1
	BLZ	BTS11		; branch if no
	LAC	MARK		; yes. mark is carrier
	SACL	TMP
	LAC	SPACE		; space is noise
	SACL	STS
	B	BTS13
;
BTS11	LAC	SPACE		; does space dominate
	SUB	MARK,1
	BLZ	BTS12		; branch if no
	LAC	SPACE		; yes. space is carrier
	SACL	TMP
	LAC	MARK		; mark is noise
	SACL	STS
	B	BTS13
;
BTS12	ZAC			; undecided. carrier is zilch
	SACL	TMP		; noise is both
	LAC	MARK
	ADD	SPACE
	SACL	STS
BTS13
;
; We shift the carrier and noise samples through a classifier register
; and extract the max carrier and min noise samples.
;
	LAR	AR1,NOIPTR	; get classifier pointer
	LAC	TMP		; stuff signal
	SACL	*+
	LAC	STS		; stuff noise
	SACL	*+
	LRLK	AR0,DELNOI+BINSIZ ; wrap pointer
	CMPR
	BBZ	BTS21
	LRLK	AR1,DELNOI
BTS21	SAR	AR1,NOIPTR	; update classifier pointer
	LALK	8000h		; initialize for max search
	SACL	TMP
	LALK	7FFFh
	SACL	STS
	LRLK	AR1,DELNOI
	LRLK	AR0,(BINSIZ/2)-1
BTS22	LAC	*+		; is this max carrier
	SUB	TMP
	BLEZ	BTS23		; branch if no
	ADD	TMP		; yes. remember that
	SACL	TMP
BTS23	LAC	*+		; is this min noise
	SUB	STS
	BGEZ	BTS24		; branch if no
	ADD	STS		; yes. remember that
	SACL	STS
BTS24	LARP	AR0		; jog to next sample
	BANZ	BTS22,*-,AR1
	LAC	TMP		; save for mark/space LED reference
	SACL	LEDREF
	LAC	TMP,1		; scale and square carrier
	SACL	TMP
	LT	TMP
	MPY	TMP
	PAC
	SACH	TMP
	LAC	STS,1		; scale and square noise
	SACL	STS
	LT	STS
	MPY	STS
	PAC
	SACH	STS
;
; We integrate the carrier and noise sample squares over the last three
; characters, or 192 samples (CARSIZ*3). The carrier distance is the
; carrier energy minus the noise energy over these samples.
;
	LAR	AR1,CARPTR	; get integrator pointer
	PSHD	TMP		; carrier square to shift register
	LAC	TMP,8
	SUB	*,8
	POPD	*+
	ADDH	CARITG		; integrate carrier
	ADDS	CARITG+1
	SACH	CARITG
	SACL	CARITG+1
	PSHD	STS		; noise square to shift register
	LAC	STS,8
	SUB	*,8
	POPD	*+
	ADDH	NOSITG		; integrate noise
	ADDS	NOSITG+1
	SACH	NOSITG
	SACL	NOSITG+1
	LRLK	AR0,DELCAR+(2*CARSIZ) ; wrap pointer
	CMPR
	BBZ	BTS31
	LRLK	AR1,DELCAR
BTS31	SAR	AR1,CARPTR	; update integrator pointer

	LAC	CARITG		; peek a boo
	SACL	PEEK1
	LAC	NOSITG		; peek another boo
	SACL	PEEK2

	LAC	NOSITG		; compute carrier distance
	SUB	CARITG
	ADD	CTHRESH		; set carrier gate
	SUBK	1<<4		; (bottom fisher)
	SACL	CSCOPE
BTSEND	RET

;
; RTTY encoder processing
;
; This routine is called at each sample decimate, which occurs at eight
; times the baud rate. It produces a Baudot character with one start bit
; and 1.5 stop bits for a total of 7.5 bits.
;
; Variables
; OUTBCB	buffer structure
; XBTCTR	bit (sample) counter
; XSMCTR	sample counter
; KEY		encoder shift register
; RTSCTR	RTS counter
;
; The 16-bit output waveform (KEY) can contain two characters, one each
; in the LSB and MSB bytes. For transmission, the word is shifted left
; one bit to make the start (space) bit of each byte and the high order
; two bits of each byte are set to make the stop (mark) bits. The word
; is then right shifted by one bit for each bit of transmission, but
; stopped midway in the eighth bit. After the last character is sent,
; the output is held at mark for one character to avoid a garble on the
; transmit/receive switch.
;
DSPOUT	LARP	AR1		; AR1 means business
	LAC	XBTCTR		; is output in progress
	BZ	XMT12		; branch if no
	SUBK	1		; yes. is this the last sample
	SACL	XBTCTR
	BNZ	XMT18		; branch if no
XMT12	LAC	RTSCTR		; yes. is this RTS delay
	BZ	XMT11		; branch if no
	SUBK	1		; yes. downdate the counter
	SACL	RTSCTR
	BIT	MODE,SYNMOD	; is sync mode enabled
	BBZ	XMT13		; branch if no
	B	XMT15		; yes. send diddle
;
XMT11	LAC	KEY		; is another char waiting
	SFR
	SBLK	0FFC0h
	BNZ	XMT17		; branch if yes
	LRLK	AR1,OUTBCB	; no. is something hiding in the buffer
	CALL	GETBUF
	BNZ	XMT16		; branch if yes
	BIT	STATUS,CTSSTA	; no. is this transmit mode
	BBZ	XMTEND		; branch if no
	BIT	STATUS,RTSSTA	; yes. is RTS enabled
	BBNZ	XMT14		; branch if yes
	CALL	FLUSH		; no. flush buffers
XMT13	LALK	00FFh		; send mark-hold
	B	XMT17
;
XMT14	BIT	MODE,SYNMOD	; is sync mode enabled
	BBZ	XMTEND		; branch if no
XMT15	LACK	SYN		; yes. send diddle (LTRS)
XMT16	CALL	ASC2BDT		; translate to Baudot (maybe two chars)
	SFL			; frame with start and stop bits
XMT17	ORK	0C0C0h
	SACL	KEY
	LACK	(7*BINSIZ)+(BINSIZ/2) ; initialize counters
	SACL	XBTCTR
	LACK	BINSIZ
	SACL	XSMCTR
	B	XMTEND
;
XMT18	LAC	XSMCTR		; is this a bit time
	SUBK	1
	BGZ	XMT20		; branch if no
	LAC	KEY		; yes. get next output bit
	SFR
	SACL	KEY
	LACK	BINSIZ		; reset sample counter
XMT20	SACL	XSMCTR
XMTEND	RET

;
; RTTY decoder processing
;
; At this point the last eight bits (64 samples) are in the survivor
; shift register. The oldest bit in the register is the stop bit of the
; previous character, the next oldest bit is the start bit of the
; current character, and the youngest is the stop bit of this character.
; A survivor is defined as the vector of samples beginning in the oldest
; bit and extending through each bit in turn to the youngest bit. There
; are eight survivors. one for each sample in the oldest bit. The
; problem here is to identify whether the oldest sample in the start bit
; begins a character and which of the eight survivors represents the
; most likely transmitted signal.
;
; The maximum difference between the mark and space channels over the
; last eight bits is called the signal range. The midpoint of this range
; is the slice level. If a survivor sample is above the slice level, it
; is a mark; otherwise, it is a space. The interval since the last start
; bit is called the frame time. In order for a character to be
; recognized, the frame time must be at least seven bits. For signals
; with stop bits greater than one unit, we expect the frame time to be
; that much longer; however, in order to affect the frame time estimate
; used by the phase-lock loop, it must not be longer than eight bits.
;
; The exponential average of the max survivor distance is called the
; autostart signal. The program operates in asynchronous mode until it
; locks on a character stream and the autostart signal has increased
; above a threshold. If so, and if enabled, the start of a character is
; determined by the phase-lock loop.
;
; Variables
; SYNCTR	RTTY sync counter (2)
; SYNDEL	frame interval
; BITCTR	bit counter
; SRVCTR	strobe counter
; XPHASE	phase error signal
; ERSDST	erasure distance
; AUTSIG	autostart signal
; SQRACC	(2)sum of squares
;
SSTOP	LARP	AR1		; AR1 means business
	LAC	SYNCTR		; increment sync counter
	ADLK	1<<8
	SBLK	(FRMSIZ+BINSIZ)<<8 ; did it overflow
	BLEZ	CYC11		; branch if no
	ZAC			; yes. clamp at max
CYC11	ADLK	(FRMSIZ+BINSIZ)<<8	
	SACL	SYNCTR
	ZAC			; initialize erasure distance
	SACL	ERSDST
	SACL	AUTSIG
	BIT	STATUS,SYNSTA	; is this sync mode
	BBZ	CYC12		; branch if no
;
; RTTY synchronous mode. In this mode the program has phase-locked to
; the start-bit transition. The decoder operates as a gated receiver,
; with the PLL signal derived only from the samples corresponding to
; start-bit intervals. We assume that the program has already refined
; the frame time in RTTY asynchronous mode. 
;
; Analysis: The VCO is controlled by a 32-bit counter (SYNCTR) which
; counts the baud interval with the decimal point to the left of bit 12
; in the MSW, so the LSB of the MSW counts in units of 1/4096 = 3.26 us.
; Therefore, at 75 baud and a delta f/f of 1 Hz, or 175.4 us in each
; period, the VCO gain is .01859 cycle/V-s. The phase detector gain is
; LIMIT/cycle = 1e4 V/cycle. Thus the loop gain is the product .01859 *
; 1e4 = 185.9. For optimum transient (Butterworth) response, the product
; of two times the time constant times the loop gain should be near
; unity. For a time constant of 1 s, which corresponds to an averaging
; factor of 1/7.5 s (three bits right shift), the product is 185.9.
; Therefore, the loop gain must be reduced by a like amount (7 bits
; right shift). With these constants, the loop lock range measures 6.0%
; and the capture range 4.0%. Since the sample rate is determined by the
; modulus of the counter, which does not affect the loop gain, the
; transient behavior is independent of baud rate.
; 
	ZALH	SYNCTR		; sync mode. is sample time too early
	SUBH	SYNDEL
	BLZ	CYCEND		; branch if yes
	ADDS	SYNCTR+1	; no. update sample counter
	ADD	XPHASE,16-4-SCONST
	SACH	SYNCTR
	SACL	SYNCTR+1
	LRLK	AR1,SRVBGN+(7*BINSIZ)-1 ; extract phase detector signal
	LAC	SLICE
	SUB	*
	SACL	TMP
	ZALR	XPHASE		; update sample clock phase
	ADD	TMP,16-SCONST
	SUB	XPHASE,16-SCONST
	SACH	XPHASE
	B	CYC21		; continue to select survivor
;
; RTTY asynchronous mode. In this mode characters may arrive at any
; time. The following little bit of paranoia insists that (a) the
; carrier gate is unblocked, (b) in the last eight bit times, the oldest
; bit (stop bit of the previous character) is a mark, (c) the next
; oldest bit (start bit of the current character) is a space, and (d)
; the youngest bit (stop bit of the current character) is a mark. These
; steps help to resist false starts due to impulse noise and fading
; and,specially, resist garbles
; due to noise.
;
CYC12	LAC	SYNCTR		; async mode. is sample time too early
	SBLK	FRMSIZ<<8
	BLZ	CYCEND		; branch if yes
	LAC	CSCOPE		; is carrier gate unblocked
	BGEZ	CYC78		; branch if yes
	LRLK	AR1,SRVBGN+BINSIZ-1 ;no. is stop bit mark
	LAC	*
	SUB	SLICE
	BLZ	CYC78		; branch if no
	ADRK	6*BINSIZ	; yes. is start bit space
	LAC	*
	SUB	SLICE
	BGZ	CYC78		; branch if no
	ADRK	BINSIZ		; yes. is last stop bit mark
	LAC	*
	SUB	SLICE
	BLZ	CYC78		; branch if no
;
; We have found the beginning of a valid character. The phase error is
; the signal at the oldest sample in the start bit. It is exponentially
; averaged to form the error signal used to drive the PLL. We compute
; the distance for each survivor according to the Viterbi algorithm.
; Note that the distance here is the negative of the Viterbi distance,
; in that the distance increases with increasing probability. The loop
; computes the distance for each survivor ending in the stop bit. We
; discard all survivors that do not show mark in the stop bit and space
; in the start bit. This is the basis of the MAP claim, since of the 128
; seven-bit words that might be received, we know that those with zero
; start bit and one stop bit have zero a-priori probability. 
;
CYC21	LRLK	AR1,SRVBGN	; initialize for survivor search
	LARK	AR2,BINSIZ-1
CYC22	LAC	*		; is stop bit mark
	SUB	SLICE
	BGEZ	CYC23		; branch if yes
	ADRK	6*BINSIZ	; no. kill survivor
	B	CYC24
;
CYC23	LAC	*,16-2		; compute distance for stop bit
	SUB	NEGMAX,16-2
	SACH	TMP
	ADRK	BINSIZ
	LAC	*,16-2		; add distance for data bit 5
	SUB	SLICE,16-2
	ABS
	ADDH	TMP
	SACH	TMP
	ADRK	BINSIZ
	LAC	*,16-2		; add distance for data bit 4
	SUB	SLICE,16-2
	ABS
	ADDH	TMP
	SACH	TMP
	ADRK	BINSIZ
	LAC	*,16-2		; add distance for data bit 3
	SUB	SLICE,16-2
	ABS
	ADDH	TMP
	SACH	TMP
	ADRK	BINSIZ
	LAC	*,16-2		; add distance for data bit 2
	SUB	SLICE,16-2
	ABS
	ADDH	TMP
	SACH	TMP
	ADRK	BINSIZ
	LAC	*,16-2		; add distance for data bit 1
	SUB	SLICE,16-2
	ABS
	ADDH	TMP
	SACH	TMP
	ADRK	BINSIZ
	LAC	*		; is start bit space
	SUB	SLICE
	BGZ	CYC24		; branch if no
	LAC	POSMAX,16-2	; yes. add distance for start bit
	SUB	*,16-2
	ADDH	TMP
	SUBH	ERSDST		; is this max survivor
	BLEZ	CYC24		; brunch if no
	ADDH	ERSDST		; yes. remember it
	SACH	ERSDST
	SAR	AR1,SRVCTR
CYC24	SBRK	(6*BINSIZ)-1	; point to the next survivor
	LARP	AR2		; AR2 is repeat counter
	BANZ	CYC22,*-,AR1	; AR1 is survivor table pointer
	LALK	32768*4*2/9	; scale erasure distance
	SACL	STMP1
	LT	ERSDST
	MPY	STMP1
	PAC
	SACH	ERSDST
	SACH	AUTSIG
;
; We have found the maximum distance survivor, which allegedly
; represents the bit string of a character, but we don't know that until
; the signal quality has been determined. We compute the RMS signal
; normalized distance and autostart. AR1 points to the start bit of the
; max survivor at this point.
;
	LAR	AR1,SRVCTR	; restore max survivor pointer
	LALK	SRVBGN+(7*BINSIZ) ; initialize for scope display
	SUB	SRVCTR
	SACL	SRVCTR
	LACK	7
	SACL	BITCTR
	ZAC			; initialize for survivor scan
	SACL	TMP
	SACH	SQRACC
	SACL	SQRACC+1
	LALK	24770		; 2*32768/sqrt(7) scale factor
	SACL	STMP2
	LRLK	AR0,7-1
	SBRK	BINSIZ*6
CYC80	LAC	SLICE		; compute slice level
	SUB	*
	ROL			; shift in data bit
	LAC	TMP
	ROL
	SACL	TMP
	LT	*		; adjust for scaling
	MPY	STMP2
	PAC
	SACH	STMP1
	LT	STMP1		; square and add
	MPY	STMP1
	PAC
	ADDH	SQRACC
	ADD	SQRACC+1
	SACH	SQRACC
	SACL	SQRACC+1
	ADRK	BINSIZ		; to next bit
	LARP	AR0		; are all samples processed
	BANZ	CYC80,*-,AR1	; branch if no
	BIT	STATUS,SYNSTA	; yes. is this sync mode
	BBNZ	CYC75		; branch if yes
;
; The (unnormallized) erasure distance (ERSDST) and autostart distance
; (AUTSIG) have been computed and the character decoded and checked for
; correct format. If sync mode, we are done here; if async mode, we are
; looking to start the character. If so, the erasure distance must
; exceed the threshold; otherwise, we just romp through the autostart
; routine once per character time.
;
	LRLK	AR1,U_ADDR+SQRACC ; extract square root
	CALL	SQRT
	SACL	STS
	ZALH	ERSDST		; normalize distance
	RPTK	15-1
	SUBC	STS
	SACL	STMP1
	LAC	ATHRESH		; is distance ok
	SUB	STMP1
	BLZ	CYC74		; branch if yes
CYC78	LAC	CHRCTR		; no. is this end of char interval
	SUBK	1
	SACL	CHRCTR
	BGZ	CYCEND		; branch if no
	LALK	FRMSIZ+BINSIZ	; yes. reset sample counter
	SACL	CHRCTR
	CALL	ASTART		; manufacture autostart signal
	B	CYCEND
;
; We have found a valid character (!). We measure the intercharacter
; interval and determine the frame time as the exponential average of
; the interval since the last start bit. The intent is to provide a
; seamless transition to synchronous mode. In order for the frame time
; to be updated, the intercharacter interval must fall within a window
; of four samples centered in the eighth bit.
;
CYC74	LAC	SYNCTR		; is sample time in range
	SBLK	(FRMSIZ+2)<<8
	BLZ	CYC76		; branch if no
	SBLK	(BINSIZ/2)<<8
	BGEZ	CYC76		; branch if no
	ZALR	SYNDEL		; yes. update the frame estimate
	ADD	SYNCTR,TCONST
	SUB	SYNDEL,TCONST
	SACH	SYNDEL
	B	CYC73
;
CYC76	ZAC			; sample time outrange. zap autostart
	SACL	AUTSIG
CYC73	ZAC			; set up for the next frame
	SACL	SYNCTR
	SACL	XPHASE
CYC75	LALK	FRMSIZ+BINSIZ	; reset sample counter
	SACL	CHRCTR
	CALL	ASTART		; manufacture autostart signal

	LALK	LIMIT*2		; make char strobe for scope
	SACL	SSYNC

	LAC	TMP		; align and mask char
	SFR
	ANDK	1Fh
	CALL	INCHAR		; process character
CYCEND	RET

;
;
; SITOR encoder processing
;
; This routine is called at each bit interval of 10 ms. It produces a
; 14-bit CCIR 476 code consisting of a new character followed by an old
; character delayed by five characters. CCIR 476 characters consists of
; four bits one and three bits zero for a total of seven bits.
;
; Variables
; OUTBCB	buffer structure
; FECCHR	character buffer
; KEY		encoder shift register
; XBTCTR	bit counter
; RTSCTR	RTS counter
;
; The implementation uses a three-word shift register, of which the
; least significant 14 bits represent the CCIR codeword. The least
; signficant seven bits of the first word represent a new character,
; while the most significant seven bits represent the repetition of the
; fifth character in the past. The figure shows the progression of the
; sequence "abc", where the least significant seven bits in all but the
; first word are not used. The CCIR 476 sync codeword is RQ-ALF.
;
;	 +---+---+   +---+---+   +---+---+   +---+---+   +---+---+
; word 3 |ALF|   |   | a |   |   | b |   |   | c |   |   |ALF|   |
;	 +---+---+   +---+---+   +---+---+   +---+---+   +---+---+
; word 2 |ALF|   |   |ALF|   |   | a |   |   | b |   |   | c |   |
;	 +---+---+   +---+---+   +---+---+   +---+---+   +---+---+
; word 1 |ALF|RQ |   |ALF| a |   |ALF| b |   | a | c |   | b |RQ |
;	 +---+---+   +---+---+   +---+---+   +---+---+   +---+---+
;
; Note that a 14-bit CCIR codeword is 140 ms in length at 100 baud (10
; ms per bit), while a standard 7.5-bit Baudot character is 150 ms in
; length at 50 baud (20 ms per bit). Thus, an unrestrained CCIR
; transmitter gains 10 ms per character. To avoid overruning the
; receiver, the transmitter inserts five CCIR RQ-ALPHA sync codewords
; after 70 non-sync codewords have been transmitted. This 10.5-s
; sequence continues indefinately as long as characters remain to be
; transmitted.
;
SITOUT	LARP	AR1		; AR1 means business
	LAC	KEY		; shift next bit
	SFR
	SACL	KEY
	LAC	XBTCTR		; is this a char time
	SUBK	1
	SACL	XBTCTR
	BGZ	STXEND		; branch if no
	LAC	RTSCTR		; yes. is this RTS delay
	BZ	STX11		; branch if no
	SUBK	1		; yes. downdate the counter
	SACL	RTSCTR
	B	STX12
;
STX11	LAC	XCHCTR		; is sync interval complete
	SUBK	1
	SACL	XCHCTR
	BGZ	STX11A		; branch if no
	LACK	70		; yes. force a sync sequence
	SACL	XCHCTR
	LACK	5
	SACL	RTSCTR
	B	STX12

STX11A	LAC	FECCHR,16-8	; is there another char
	SACH	FECCHR
	LAC	FECCHR
	BNZ	STX14		; branch if yes
	LRLK	AR1,OUTBCB	; no. is something hiding in the buffer
	CALL	GETBUF
	BNZ	STX13		; branch if yes
	BIT	STATUS,CTSSTA	; no. is this transmit mode
	BBZ	STX12		; branch if no
	BIT	STATUS,RTSSTA	; yes. is RTS enabled
	BBNZ	STX12		; branch if yes
	CALL	FLUSH		; no. flush buffers
STX12	LALK	RQ		; send sync code
	OR	KEY+1
	SACL	KEY
	LAC	KEY+2
	SACL	KEY+1
	LALK	ALPHA<<7
	SACL	KEY+2
	B	STX21
;
STX13	CALL	ASC2BDT		; buffer hiding. translate to Baudot
	SACL	FECCHR		; (note there could be two chars here)
STX14	CALL	BDT2CCR		; translate to CCIR
;
; Assemble characters for transmission. Each transmission consists of a
; new character plus an old one saved from five characters previously.
; This code forces a five-character sync sequence every 70 characters,
; which is crude, but effective.
;
	OR	KEY+1		; get previous char for repetition
	SACL	KEY
	LAC	KEY+2
	SACL	KEY+1
	LAC	KEY,7		; save current char for repetition
	SACL	KEY+2
STX21	LACK	14		; assemble 14 bits
	SACL	XBTCTR
STXEND	RET

;
; SITOR decoder processing
;
; At this point the last eight bits (64 samples) are in the survivor
; shift register in order to calculate the carrier and noise estimates,
; but we need only the first bit (eight samples) to determine the symbol
; phase and decoder signal. The quadrature-phase samples are used to
; derive the VCO signal for the PLL, which is implemented by modulating
; the bit sample interval. The PLL controls the symbol phase independent
; of character synchronization.
;
; CCIR 476 characters are transmitted twice, once in the normal position
; and again in the repeat position (delayed by five characters). In the
; normal case, the original and repeat characters are the same, so can
; be vector summed before correlation. We do this using a six-character
; (42-bit vector) shift register. When no characters for transmission
; are available, or in order to provide fill, a special synchronization
; sequence is used consisting of the CCIR 476 characters RQ followed by
; ALPHA; therefore, this sequence must be correlated separately.
;
; Variables
; TORDLD/P	received bit vector (six seven-bit characters)
; SLICE		comparator slice level
; BITCTR	bit counter
; SMPCTR	strobe counter
; COUNT		baud interval counter
; XPHASE	phase error signal
; ERSDST	erasure distance
; TORPTR	SITOR sync delay line pointer
; TORACP	SITOR sync accumulator pointer
;
SITOR	LARP	AR1		; AR1 means business
	LAC	SMPCTR		; is it time for the next bit
	SUBK	1
	SACL	SMPCTR
	BGZ	SITEND		; branch if no
	LACK	BINSIZ		; yes. restore decimation counter
	SACL	SMPCTR
;
; If the signal polarity at the beginning and end of the bit is
; opposite, the midpoint of the bit is likely a good estimate of the
; symbol phase. This is used to update the PLL.
;
; Analysis: The VCO is controlled by two-word counter (COUNT) in
; microseconds and fraction, with a period of 10000 us. Therefore, for
; delta f = 1 Hz, or 1000000/10000 = 100, the VCO gain is 1/100 = .01
; Hz/V. The phase detector gain is LIMIT = 10000 V/Hz-s. Thus the loop
; gain is the product .01 * 10000 = 100. For optimum transient
; (Butterworth) response, the product of two times the time constant
; times the loop gain should be near unity. For a time constant of 0.5
; s, which corresponds to an averaging factor of BAUD/2 = 50, the
; product is 100. Therefore, the loop gain must be reduced by a like
; amount. With these constants, the loop lock range measures 3.4% and
; the capture range 1.3%. As the time constant (MCONST) is increased,
; the loop gain, lock range and capture range are reduced by the same
; factor.
;
	LRLK	AR1,SRVBGN	; is new bit positive
	LAC	*
	SUB	SLICE
	BLZ	SIT11		; branch if no
	ADRK	BINSIZ		; yes. is last bit also positive 
	LAC	*
	SUB	SLICE
	BGZ	SIT13		; branch if yes
	SBRK	BINSIZ/2	; no. extract midpoint sample
	LAC	*
	SUB	SLICE
	B	SIT12
;
SIT11	ADRK	BINSIZ		; new bit is negative
	LAC	*		; is last bit also negative
	SUB	SLICE
	BLZ	SIT13		; branch if yes
	SBRK	BINSIZ/2	; no. extract midpoint sample
	LAC	SLICE
	SUB	*
SIT12	SACL	TMP
	ZALR	XPHASE		; update symbol phase
	ADD	TMP,16-MCONST
	SUB	XPHASE,16-MCONST
	SACH	XPHASE
SIT13	ZALH	COUNT		; update VCO frequency
	ADDS	COUNT+1
	SUB	XPHASE,16-MCONST-2
	SACH	COUNT
	SACL	COUNT+1
;
; To determine character sync, we correlate the characters in the normal
; and repeat position of the CCIR sequence RQ-ALPHA. The signal is
; continously measured for each of 14 bit positions of five repetitions
; of the sequence, for a total of 70 bits. The distance is normalized
; relative to the RMS of all samples in the shift register. If the
; distance exceeds a threshold (ATHRESH), character phase is
; established. The threshold is chosen to minimize the false-alarm rate,
; yet maximize the probability of correct bit synchronization with the
; 700-ms restrain sequence (five repetitions of RQ-ALPHA), normally
; transmitted every 70 characters) or at the end of a line.
;
; The cyclic autocorrelation function of the CCIR 476 RQ-ALPHA sequence
; is rather nice:
;
; offset 0  1  2  3  4  5  6  7  8  9 10 11 12 13
; value	14  6 -2 -2 -2 -2 -2 -2 -2 -2 -2 -2 -2  6
;
; We have to be a bit sneaky here, since the retrain sequences are
; embedded in ordinary text. We have to skew the correlation of the
; ALPHA two codewords later than the RQ, to wit (* = unrelated char):
;
;    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+
; ...|RQ   * |RQ   * |RQ  ALF|RQ  ALF|RQ  ALF| *  ALF| *  ALF|...
;    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+
;
; Correlate RQ-ALPHA codeword and calculate signal and signal square
;
	CNFD			; map bit vector to data memory
	LRLK	AR1,SRVBGN
	LAC	*		; get bias-corrected bit sample
	SUB	SLICE
	LRLK	AR1,TORDLD+TORSIZ-2 ; shift bit vector delay line right
	RPTK	TORSIZ-2 
	DMOV	*-
	MAR	*+
	SACL	*		; insert new bit in bit vector
	CNFP			; map bit vector to program memory
	ZAC			; calculate signal distance
	MPYK	0
	LRLK	AR1,TORTAB+(7*(CCTALF-CCIR))
	RPTK	7-1
	MAC	TORDLP,*+	; add distance for ALPHA in rpt position
	LRLK	AR1,TORTAB+(7*(CCTRQ-CCIR))
	RPTK	7-1
	MAC	TORDLP+(7*5),*+	; add distance for RQ in orig position
	APAC
	SACH	ERSDST
	CALL	SUMSQ		; compute signal square
;
; Integrate signal and signal square over 14 bits and five codewords
;
	LALK	32768/5		; scale signal and signal square
	SACL	STS
	LT	ERSDST
	MPY	STS
	PAC
	SACH	TMP
	LT	SQRACC
	MPY	STS
	PAC
	SACH	STS
	LAR	AR1,TORACP	; AR1 is accumulator pointer
	LAR	AR2,TORPTR	; AR2 is delay line pointer
	LARP	AR2
	PSHD	STS		; filter signal square
	ZALH	STS
	SUBH	*
	POPD	*+
	LARP	AR1
	ADDH	*		; accumulate signal square
	SACH	*+
	SACH	STS
	LARP	AR2
	PSHD	TMP		; filter signal
	ZALH	TMP
	SUBH	*
	POPD	*+
	LRLK	AR0,TORSTR+(14*2*5) ; wrap delay line pointer
	CMPR
	BBZ	SIT30
	LRLK	AR2,TORSTR
SIT30	SAR	AR2,TORPTR 
	LARP	AR1
	ADDH	*		; accumulate signal
	SACH	*+
	BGEZ	SIT22		; toss out the weaklings
	ZAC
SIT22	SACH	TMP
	LRLK	AR0,TORACC+(14*2) ; wrap accumulator pointer
	CMPR
	BBZ	SIT31
	LRLK	AR1,TORACC
SIT31	SAR	AR1,TORACP
	ZALH	STS 		; extract square root
	SACH	SQRACC
	SACL	SQRACC+1
	LRLK	AR1,U_ADDR+SQRACC
	CALL	SQRT
	SACL	STS
	ZALH	TMP		; normalize distance
	RPTK	15-1
	SUBC	STS
	SACL	STS
	LAC	ETHRESH		; is erasure threshold exceeded
	SUB	STS

;	SACL	PEEK1

	BGEZ	SIT21		; branch if no
	ZAC			; yes. reset phase
	SACL	BITCTR
SIT21	LAC	BITCTR		; is this end of scan
	BNZ	SITEND		; branch if no
;
; Character processing. Fiddle with code conversion and distance. Note
; that we have already computed the distance for the RQ-ALPHA sync code.
;
	LALK	LIMIT*2		; make char strobe for scope
	SACL	ACCUM
	LACK	14		; initialize for next char
	SACL	BITCTR
	LACK	BINSIZ
	SACL	SRVCTR
	CALL	CCR2BDT		; convert to Baudot and find distance
	SUBK	20h		; is this a valid Baudot char
	BGEZ	SITEND		; branch if no
	ADDK	20h		; yes. outfumble the dang thing
	CALL	INCHAR
SITEND	RET

;
; The following two routines implement two different synchronous
; channel filter/detector/lowpass filters. These routines process the
; input filtered and limited signal to produce baseband mark and space
; signals. Each routine is optimized for a different mode. In the
; synchronous filters, the input signal is multiplied by sin/cos
; functions and the I and Q quadrature signals are processed in four
; identical lowpass filters. The baseband signal is then computed as the
; square root of the sum of the squares of the I and Q channels. All
; routines have a common calling sequence and return the basband signal
; in the AC.
;
; Variables
; AR1		channel structure pointer
; AR2		IIR delay line pointer (if needed)
; FLTSIG	input signal
; MFSIZE	size of matched filter (samples * 4)
; MFGAIN	matched filter scale
; GAIN		filter gain
; returns	filtered signal
;
; Matched filter
;
; This routine implements a filter/detector optimized for high
; selectivity. It uses synchronous matched filters tuned to the baud
; interval. A single interleaved shift register is used for the I and Q
; channels and mark and space frequencies.
;
MFILT	LAC	*+		; (-increment)
	ADD	*		; (phase) did pointer underflow
	BGEZ	MFL01		; branch if no
	ADLK	PI2		; yes. adjust pointer
MFL01	SACL	*		; save and advance structure pointer
	LAC	*+,8		; scale for table offset
	SACH	STS
;
; I channel
;
	LAC	STS		; get sin argument
	ADLK	SINTAB
	TBLR	TMP
	LT	TMP		; sin * signal
	MPY	FLTSIG
	LTP	GAIN		; adjust gain (256+-)
	SACH	TMP
	MPY	TMP
	LTP	MFGAIN
	SACH	TMP,7
	LARP	AR0		; AR0 is MF delay line pointer
	LAR	AR0,MFPTR
	PSHD	TMP		; add new value
	ZALH	TMP
	SUBH	*		; subtract old value
	POPD	*+,AR1		; insert new value in mf
	SACH	TMP
	SAR	AR0,MFPTR
	MPY	TMP		; scale for unity gain
	PAC
	ADDH	*+		; add to accumulator
	ADDS	*-
	SACH	*+
	SACL	*+
;
; Q channel
;
	LAC	STS		; get cos argument
	ADLK	30+SINTAB
	TBLR	TMP
	LT	TMP		; cos * signal
	MPY	FLTSIG
	LTP	GAIN		; adjust gain (256+-)
	SACH	TMP
	MPY	TMP
	LTP	MFGAIN
	SACH	TMP,7
	LARP	AR0		; AR0 is MF delay line pointer
	LAR	AR0,MFPTR
	PSHD	TMP		; add new value
	ZALR	TMP
	SUBH	*		; subtract old value
	POPD	*+,AR1		; insert new value in mf
	SACH	TMP
	SAR	AR0,MFPTR
	MPY	TMP		; scale for unity gain
	PAC
	ADDH	*+		; add to accumulator
	ADDS	*-
	SACH	*+
	SACL	*+
	CALL	SQSQR		; square, sum and root
	RET

;
; Butterworth IIR filter
;
; This routine implements a filter/detector filter optimized for quasi-
; linear detector frequency response. It uses synchronous, second-order
; Butterworth IIR lowpass filters. The bandwidth is determined by the
; coefficients loaded in block B0P.
;
IFILT	LAC	*+		; (-increment)
	ADD	*		; (phase) did pointer underflow
	BGEZ	IFL01		; branch if no
	ADLK	PI2		; yes. adjust pointer
IFL01	SACL	*		; save and advance structure pointer
	LAC	*+,8		; scale for table offset
	SACH	STS
;
; I channel
;
	LAC	STS		; get sin argument
	ADLK	SINTAB
	TBLR	TMP
	LT	TMP		; sin * signal
	MPY	FLTSIG
	LTP	GAIN		; adjust gain (128/512+-)
	SACH	TMP
	MPY	TMP
	PAC
	LARP	AR2		; AR2 is IIR delay line pointer
	MAR	*+		; (AR1->z^-1)
	MPYK	0		; initialize sum and product registers
	MACD	LPFCOP,*-	; process a coefficients
	MACD	LPFCOP+1,*
	APAC
	SACH	*,5-1		; normalize
	MPYK	0		; initialize sum and product registers
	ZAC
	RPTK	3-1		; process b coefficients
	MAC	LPFCOP+3,*+
	APAC
	LARP	AR1		; AR1 is structure pointer
	SACH	*+
	SACL	*+
;
; Q channel
;
	LAC	STS		; get cos argument
	ADLK	30+SINTAB
	TBLR	TMP
	LT	TMP		; sin * signal
	MPY	FLTSIG
	LTP	GAIN		; adjust gain (128/512+-)
	SACH	TMP
	MPY	TMP
	PAC
	LARP	AR2		; AR2 is IIR delay line pointer
	MAR	*+		; (AR1->z^-1)
	MPYK	0		; initialize sum and product registers
	MACD	LPFCOP,*-	; process a coefficients
	MACD	LPFCOP+1,*
	APAC
	SACH	*,5-1		; normalize
	MPYK	0		; initialize sum and product registers
	ZAC
	RPTK	3-1		; process b coefficients
	MAC	LPFCOP+3,*+
	APAC
	LARP	AR1		; AR1 is structure pointer
	SACH	*+
	SACL	*+
	CALL	SQSQR		; square, sum and root
	SACL	TMP

;
; Process the baseband signal in a matched filter matched to the baud
; rate.
;
	LARP	AR0		; AR0 is MF delay line pointer
	LAR	AR0,MFPTR
	PSHD	TMP
	ZALH	TMP
	SUBH	*		; subtract old value
	POPD	*,AR1		; insert new value in mf
	SACH	TMP
	LACK	2		; increment filter pointer
	ADD	MFPTR
	SACL	MFPTR
	LT	MFGAIN
	MPY	TMP		; scale for unity gain
	PAC
	ADRK	2		; add to accumulator
	ADDH	*+
	ADDS	*-
	SACH	*+
	SACL	*-
	LAC	*		; return high-order sum
	RET

;
; Subroutine to square, sum and root. AR1 points just past the
; accumulator
;
SQSQR	SBRK	4		; point to high order I channel sum
	MPYK	0		; initialize sum and product registers
	ZAC
	SQRA	*		; I^2
	ADRK	2
	SQRA	*		; Q^2
	ADRK	2
	APAC			; sum
	SACH	*+
	SACL	*-
	CALL	SQRT		; square root
	RET
;
; Find square root using Newton-Raphelson (3 iterations). Initial
; estimate for x assumes the signal is at the limit threshold. AR1
; points to the square (two words); on exit, the square root is in the
; AC.
;
SQRT	LALK	LIMIT		; initial guess
	SACL	STMP1
	ZALH	*		; (c / x) iteration 1
	RPTK 	15-1
	SUBC	STMP1
	SACL	STMP2
	ZALH	STMP2		; (c / x + x)
	ADDH	STMP1
	SFR			; (1 / 2 * (c / x + x))
	SACH	STMP1
	ZALH	*		; (c / x) iteration 2
	RPTK 	15-1
	SUBC	STMP1
	SACL	STMP2
	ZALH	STMP2		; (c / x + x)
	ADDH	STMP1
	SFR			; (1 / 2 * (c / x + x))
	SACH	STMP1
	ZALH	*		; (c / x) iteration 3
	RPTK 	15-1
	SUBC	STMP1
	SACL	STMP2
	ZALH	STMP2		; (c / x + x)
	ADDH	STMP1
	SFR			; (1 / 2 * (c / x + x))
	SACH	STMP1
	LAC	STMP1
	RET

;
; Routine to decode keyboard input and interpret commands
;
; The following routine implements a crude, table-driven command decoder
; which can be used to select various system parameters, such as baud
; rate, channel spacing, etc. It operates by sampling the UART receive-
; done bit. Someday this will be driven by interrupts; meanwhile, it
; operates with interrupts disabled. Note that the UART hardware
; parameters are set up by the downloader.
;
; Variables
; INPBCB	input buffer structure
; OUTBCB	output buffer structure
; UCHAR		character temp
; UPTR		pointer temp
; UADR		address temp
; UCMD		command table pointer
; UMOD		saved command table pointer
;
GETCH	LARP	AR1		; AR1 means business
	DINT			; bolt the windows
	LDPK	B_PAGE
	LALK	UART_SEL_LSR	; select line status register
	SACL	UARTI
	OUT	UARTI,UART_CTRL
	IN	UARTI,UART_READ
	LALK	UART_LSR_RBF	; is input buffer full
	AND	UARTI
	BNZ	GET14		; branch if yes
	LDPK	U_PAGE		; no. allow fresh air
	EINT
	RET

GET14	LALK	UART_SEL_RX	; select input buffer
	SACL	UARTI
	OUT	UARTI,UART_CTRL 
	IN	UARTI,UART_READ	; input character
	LAC	UARTI
	LDPK	U_PAGE
	EINT			; allow fresh air
	ANDK	07Fh		; unparity it
	SACL	UTMP2
	SUBK	060h		; uncase it
	BLZ	GET15
	SUBK	020h
GET15	ADDK	060h
	SACL	UTMP1
	SUBK	EOT		; is char EOT
	BNZ	GET90		; branch if no
	CALL	FLUSH		; yes. flush buffers
	B	GET92
;
GET90	SUBK	ACK-EOT		; is char ACK
	BNZ	GET91		; branch if no
GET92	LALK	~(1<<RTSSTA)	; yes. kill RTS
	AND	STATUS
	SACL	STATUS
	B	GET12
;
GET91	BIT	STATUS,CTSSTA	; is this transmit mode
	BBZ	GET95		; branch if no
	LRLK	AR1,OUTBCB	; yes. send char
	LAC	UTMP1
	CALL	PUTBUF
	BIT	MODE,FDXMOD	; is this full duplex
	BBNZ	GET12		; branch if yes
	BIT	MODE,ECHMOD	; no. is digital loopback echo selected
	BBZ	GET94		; branch if no
	LRLK	AR1,INPBCB	; yes. echo char
	LAC	UTMP2
	CALL	PUTBUF
GET94	LAC	UTMP2		; was last char CR
	SUBK	CR
	BNZ	GET12		; branch if no
	LRLK	AR1,OUTBCB	; yes. send LF
	LACK	LF 
	CALL	PUTBUF
	BIT	MODE,ECHMOD	; is digital loopback echo selected
	BBZ	GET12		; branch if no
	LRLK	AR1,INPBCB	; yes. echo LF
	LACK	LF
	CALL	PUTBUF
	B	GET12
;
GET95	LRLK	AR1,INPBCB	; echo char
	LAC	UTMP1
	CALL	PUTBUF
	LAC	UTMP1		; was last char CR
	SUBK	CR
	BNZ	GET13		; branch if no
	LRLK	AR1,INPBCB	; yes. send LF
	LACK	LF 
	CALL	PUTBUF
GET13	LAC	UCMD		; initialize table pointer
	SACL	UPTR
GET10	LAC	UPTR		; get next routine address
	TBLR	UADR
	ADDK	1		; get argument
	TBLR	UTMP2
	ADDK	1		; get next command character
	TBLR	UCHAR
	ADDK	1
	SACL	UPTR
	LAC	UCHAR		; is this last entry
	BZ	GET11		; branch if yes
	SUB	UTMP1		; no. does character match
	BNZ	GET10		; branch if no
GET11	LAC	UADR		; yes. call routine
	CALA
GET12	RET
;
; Housekeeping staff
;
GETPSH	LAC	UCMD		; save command table pointer
	SACL	UMOD
	LAC	UTMP2
	SACL	UCMD
	RET
;
GETPOP	LAC	UMOD		; restore command table pointer
	SACL	UCMD
	B	GET13		; continue in command decode

;
; Following are command execution segments. For compactness, we assume
; the auxiliary register pointer is set at AR1.
;
; Restore to initial defaults
;
RESET	LARP	AR1		; (in case called directly)
	ZAC			; initialize things
	LRLK	AR1,B1D		; clear block B1 (0300-03ff)
	RPTK	100h-1
	SACL	*+
	LRLK	AR1,USRBGN	; clear external memory (0401-)
	LRLK	AR2,USREND-USRBGN-1
RST10	SACL	*+
	LARP	AR2		; AR2 is repeat counter
	BANZ	RST10,*-,AR1
	LALK	ST00		; initialize command state machine
	SACL	UCMD
	LALK	((BINSIZ/2)+FRMSIZ)<<8 ; set frame interval
	SACL	SYNDEL
	LALK	CLIMIT		; set carrier threshold
	SACL	CTHRESH
	LALK	ELIMIT		; set erasure threshold
	SACL	ETHRESH
	LALK	ALIMIT		; set autostart threshold
	SACL	ATHRESH
	LACK	5		; set limiter gain
	SACL	LGAIN
	LALK	0FFFFh		; set output mark-hold
	SACL	KEY
	LALK	U_ADDR+STS	; select monitoring point
	SACL	OUTSW
	LALK	BAUD2		; select 75 baud
	CALL	CMDB1
	LALK	SHIFT2		; select 600 Hz shift, IIR filter
	CALL	CMDM1
	CALL	TINIT		; build SITOR tables
	LALK	TORSTR		; initialize SITOR sync pointers
	SACL	TORPTR
	LALK	TORACC
	SACL	TORACP
	LALK	AUTDEL		; initialize autostart pointer
	SACL	AUTPTR
	LALK	DELCAR		; initialize carrier/noise pointers
	SACL	CARPTR
	LALK	DELNOI
	SACL	NOIPTR
	LALK	GAIN4		; reset analog gain
	SACL	AGAIN
	RET

;
; Select SITOR mode
;
CMDY	LALK	BAUD1		; select 100 baud
	CALL	CMDB1
	LALK	SHIFT8		; select 170 Hz shift, matched filter
	CALL	CMDM1
	LALK	(1<<TORMOD)+(1<<SYNMOD)	; enable SITOR
	OR	MODE
	SACL	MODE
	RET

;
; Select mode, filter and shift m{1-8}
;
CMDM	LAC	UTMP2		; get initialization table pointer
CMDM1	LRLK	AR1,MSTRUCT
	TBLR	*+		; set mark filter routine
	ADDK	1
	TBLR	*		; set mark filter frequency
	ADDK	1
	LRLK	AR1,SSTRUCT
	TBLR	*+		; set space filter routine
	ADDK	1
	TBLR	*		; set space filter frequency
	ADDK	1
	TBLR	MODE		; get mode
	ADDK	1		; set filter gain
	TBLR	GAIN
	CALL	CMDO0		; set output shift
	CNFD			; map B0 to data memory
	LRLK	AR1,B0D
	BIT	MODE,NARMOD	; is this narrowband filter
	BBNZ	CMDM2		; branch if yes
	RPTK	FIRSIZ-1	; no. use wideband coefficients
	BLKP	BPF850,*+
	RPTK	6-1
	BLKP	IIR1,*+
	CNFP			; map B0 to program memory
	B	CMD55
;
CMDM2	RPTK	FIRSIZ-1	; use narrowband coefficients
	BLKP	BPF170,*+
	RPTK	6-1
	BLKP	IIR1,*+
	CNFP			; map B0 to program memory
	B	CMD55
;
; Adjust space frequency m{+-}
;
CMDSP	LALK	-TUNEX		; m+ increase frequency
	B	CMD22

CMDSN	LALK	TUNEX		; m- reduce frequency
CMD22	LRLK	AR1,SSTRUCT+1
	ADD	*
	SACL	*
	RET
;
; Select baud rate b{1-4}
;
CMDB	LAC	UTMP2		; get initialization table pointer
CMDB1	TBLR	RANGE		; AIO samples
	ADDK	1
	TBLR	MFSIZE		; MF size
	ADDK	1
	TBLR	MFGAIN		; MF gain
CMD55	ZAC
	LRLK	AR1,MSTRUCT+2	; clear mark filter structure
	RPTK	STRSIZ-2-1
	SACL	*+
	LRLK	AR1,SSTRUCT+2	; clear space filter structure
	RPTK	STRSIZ-2-1
	SACL	*+
	LRLK	AR1,DELLPF	; clear IIR delay line
	RPTK	12-1
	SACL	*
	LRLK	AR1,DELMFR	; clear MF delay line
	LRLK	AR2,MFRSIZ-1
CMD56	SACL	*+
	LARP	AR2		; AR2 is repeat counter
	BANZ	CMD56,*-,AR1	; AR1 is MF delay line pointer
	LALK	DELMFR		; leave MF pointer in handy place
	SACL	MFPTR
	CALL	FLUSH		; flush buffers
	RET
;
; Adjust baud rate b{+-}
;
CMDBP	LAC	RANGE		; b+ increase baud interval
	ADDK	1
	SACL	RANGE
	RET
;
CMDBN	LAC	RANGE		; b- decrease baud interval
	SUBK	1
	SACL	RANGE
	RET
;
; Select output shift o{0-2}
;
CMDO0	BIT	MODE,NARMOD	; o0 is input narrow shift
	BBZ	CMDO2		; branch if no
CMDO1	LALK	-(2125*FAC)	; o1 yes. select narrow shift
	SACL	OMARK
	LALK	-(2295*FAC)
	SACL	OSPACE
	RET
;
CMDO2	LALK	-(1275*FAC)	; o2 select wide shift
	SACL	OMARK
	LALK	-(2125*FAC)
	SACL	OSPACE
	RET
;
; Select limiter gain l{0-5}
;
CMDL	LAC	UTMP2		; select limiter gain
	SACL	LGAIN
	RET
;
; Tune radio t{+-}
;
CMDTP	LALK	FREQ_UP_R1+FREQ_UP_R2 ; t+ up 50 Hz (ICOM)
	B	CMDT1
;
CMDTN	LALK	FREQ_DN_R1+FREQ_DN_R2 ; t- down 50 Hz (ICOM)
CMDT1	SACL	TUNE
	LALK	TUNEW
	SACL	TUNCTR
	RET
;
; Adjust analog gain g{0-7}
;
CMDGG	LAC	UTMP2		; g{0-7} adjust analog gain
	SACL	AGAIN
	RET
;
; Adjust filter gain g{+-}
;
CMDGP	ZALR	GAIN		; g+ decrease gain
	SUB	GAIN,16-5
	SACH	GAIN
	RET
;
CMDGN	ZALR	GAIN		; g- increase gain
	ADD	GAIN,16-5
	SACH	GAIN
	RET
;
; Select autostart gate a{0-1}
;
CMDA	LALK	~(1<<AUTMOD)	; dim tired bit
	AND	MODE
	OR	UTMP2		; light fresh bits
	SACL	MODE
	ZAC			; kill autostart counter
	SACL	AUTCTR
	RET
;
; Adjust carrier threshold c{+-}
;
CMDCP	ZALR	CTHRESH		; c+ decrease carrier threshold
	SUB	CTHRESH,16-4
	SACH	CTHRESH
	RET
;
CMDCN	ZALR	CTHRESH		; c- increase carrier threshold
	ADD	CTHRESH,16-4
	SACH	CTHRESH
	RET
;
; Adjust autostart threshold a{+-}
;
CMDAP	ZALR	ATHRESH		; a+ decrease autostart threshold
	SUB	ATHRESH,16-4
	SACH	ATHRESH
	RET
;
CMDAN	ZALR	ATHRESH		; a- increase autostart threshold
	ADD	ATHRESH,16-4
	SACH	ATHRESH
	RET
;
; Adjust erasure threshold v{+-}
;
CMDVP	ZALR	ETHRESH		; v+ decrease erasure threshold
	SUB	ETHRESH,16-4
	SACH	ETHRESH
	RET
;
CMDVN	ZALR	ETHRESH		; v- increase erasure threshold
	ADD	ETHRESH,16-4
	SACH	ETHRESH
	RET
;
; Select channel c{0-2}
;
CMDC	LALK	~((1<<MRKMOD)+(1<<SPCMOD)) ; dim tired bits
	AND	MODE
	OR	UTMP2		; light fresh bits
	SACL	MODE
	RET
;
; Select display/output d{+-, 0-9}
;
CMDD	LAC	UTMP2		; select monitor point
	SACL	OUTSW
	RET
;
; Select analog/digital loopback echo e{1-3}
;
CMDE	LALK	~((1<<FDXMOD)+(1<<ECHMOD)) ; dim tired bits
	AND	MODE
	OR	UTMP2		; light fresh bits
	SACL	MODE
	RET
;
; Select unshift-on-space u{0-3}
;
CMDU	LALK	~((1<<RUSMOD)+(1<<XUSMOD)) ; dim tired bits
	AND	MODE
	OR	UTMP2		; light fresh bits
	SACL	MODE
	RET
;
; Select radio port p{1-2}
;
CMDP1	LALK	~(1<<PORMOD)	; p1 radio port 1
	AND	MODE
	SACL	MODE
	RET
;
CMDP2	LALK	1<<PORMOD	; p2 radio port 2
	OR	MODE
	SACL	MODE
	RET
;
; Toggle bit s, i
;
CMDTOG	LALK	~(1<<SYNSTA)	; kill sync (just in case)
	AND	STATUS
	SACL	STATUS
	LAC	MODE		; toggle mode bit
	XOR	UTMP2
	SACL	MODE
	RET
;
; Begin transmission x
;
CMDX	CALL	FLUSH		; flush buffers
	LACK	RHOLD		; initialize RTS delay
	SACL	RTSCTR
	LALK	(1<<RTSSTA)+(1<<CTSSTA) ; set transmit mode
	OR	STATUS
	SACL	STATUS
	LALK	PTT_R1+PTT_R2	; light up the transmitter
	SACL	TUNE
	RET

;
; UART output watcher
;
; This routine tests if the UART output buffer is nonempty. If so it
; loads the modem control register and, if the output buffer is
; nonempty, loads the the UART output buffer.
;
; Variables
; UARTO		UART output buffer
; UARTC		UART modem control buffer
; INPBCB	input buffer structure
; 
PUTCH	LARP	AR1		; AR1 means business
	DINT			; bolt the windows
	LDPK	B_PAGE
	LALK	UART_SEL_LSR	; select line status register
	SACL	XTMP
	OUT	XTMP,UART_CTRL
	IN	XTMP,UART_READ	; is output buffer empty
	LACK	UART_LSR_THRE
	AND	XTMP
	BZ	URX10		; branch if no
	LALK	UART_SEL_MCR	; yes. select modem control register
	SACL	XTMP
	OUT	XTMP,UART_CTRL	; output modem control
	LAC	UARTC
	SACL	XTMP
	OUT	XTMP,UART_WRITE
	LALK	UART_LSR_THRE	; select output buffer
	SACL	XTMP
	OUT	XTMP,UART_CTRL
	LDPK	U_PAGE
	EINT
	LRLK	AR1,INPBCB	; are characters hiding in the buffer
	CALL	GETBUF
	DINT
	LDPK	B_PAGE
	BZ	URX10		; branch if no
	SACL	UARTO		; yes. output character
	OUT	UARTO,UART_WRITE
URX10	LDPK	U_PAGE
	EINT			; allow fresh air
	RET

;
; Input character processing utility
;
; This routine does a wonderfully intricate dance to determine whether
; to print a Baudot character, to print an erasure character "_", or to
; print nothing. It manages the autostart counter and RTTY sync mode
; bit. We stash the character in the input buffer, unless transmitting
; in digital loopback echo mode. We stash the character in the output
; buffer as well, but only if receiving in half-duplex modes or driving
; in the left-hand lane. Got that?
;
INCHAR	SACL	TMP		; park somewhere tidy
	LALK	~(1<<SYNSTA)	; provisionally killl sync
	AND	STATUS
	SACL	STATUS
	LAC	ASCOPE		; is autostart gate unblocked
	BLZ	INC17		; branch if yes
	BIT	MODE,TORMOD	; no. is this SITOR mode
	BBZ	INC18		; branch if no
	B	INCEND		; yes. ignore char
;
INC17	LALK	AHOLD		; reset autostart counter
	SACL	AUTCTR
	BIT	MODE,SYNMOD	; is sync enabled
	BBZ	INC18		; branch if no
	LALK	1<<SYNSTA	; yes. set sync mode
	OR	STATUS
	SACL	STATUS
INC18	LAC	ESCOPE		; is erasure distance ok
	BLZ	INC14		; branch if yes
	ZAC			; is suppress erasures enabled
	BIT	MODE,ERSMOD
	BBNZ	INCEND		; branch if yes
	LACK	'_'		; no. make a mark
	B	INC15
;
INC14	LAC	AUTCTR		; is autostart counter running
	BZ	INC19		; branch if no
	LALK	AHOLD		; yes. reset it
	SACL	AUTCTR
INC19	BIT	MODE,AUTMOD	; is autostart enabled
	BBZ	INC10		; branch if no
	LAC	AUTCTR		; yes. is autostart ok
	BZ	INCEND		; branch if no
INC10	LAC	TMP		; translate Baudot char to ASCII
	CALL	BDT2ASC
	BZ	INCEND		; suppress nonprint characters
INC15	SACL	TMP
	BIT	STATUS,CTSSTA	; is this digital loopback echo
	BBNZ	INC12		; branch if no
	BIT	MODE,FDXMOD
	BBZ	INC12		; branch if no
	BIT	MODE,ECHMOD
	BBNZ	INC11		; branch if yes
INC12	LRLK	AR1,INPBCB	; no. copy to input buffer
	LAC	TMP
	CALL	PUTBUF
INC11	BIT	STATUS,CTSSTA	; is this half-duplex receive
	BBNZ	INCEND		; branch if no
	BIT	MODE,FDXMOD
	BBNZ	INCEND		; branch if no
	LRLK	AR1,OUTBCB	; yes. copy to output buffer
	LAC	TMP
	CALL	PUTBUF
INCEND	RET			; declare victory

;
; Routine to put a word in a circular buffer
;
; On entry, AR1 points to the buffer structure and the data word is in
; AC. Zero words are not stored. ARP must point to AR1.
;
PUTBUF	BZ	BUF10		; brook no zeros
	LAR	AR0,*+		; get upper limit
	MAR	*+		; get put pointer
	LAR	AR2,*,AR2
	MAR	*+		; increment put pointer
	CMPR			; is it at upper limit
	LARP	AR1
	BBZ	BUF11		; branch if no
	MAR	*-		; yes. wrap to lower limit
	LAR	AR2,*+
BUF11	SAR	AR2,*,AR2	; save put pointer
	SACL	*,,AR1		; put char in circular buffer
	LAC	*+		; did the buffer overflow
	SUB	*
	BNZ	BUF10		; branch if no
	LALK	1<<ERRSTA	; yes. remember that
	OR	STATUS
	SACL	STATUS
BUF10	RET

;
; Routine to get a word from a circular buffer
;
; On entry, AR1 points to the buffer structure. On exit, the word is in
; AC or zero if the buffer is empty. ARP must point to AR1.
;
GETBUF	LAR	AR0,*+		; get upper limit
	MAR	*+		; get put pointer
	LAC	*+
	SUB	*		; get get pointer
	BZ	BUF20		; branch if equal (buffer empty)
	LAR	AR2,*,AR2	; unequal. get get pointer
	MAR	*+		; increment get pointer
	CMPR			; is it at upper limit
	LARP	AR1
	BBZ	BUF21		; branch if no
	SBRK	2		; yes. wrap to lower limit
	LAR	AR2,*
	ADRK	2
BUF21	SAR	AR2,*,AR2	; save get pointer
	LAC	*,,AR1		; get char from circular buffer
BUF20	RET

;
; Routine to get the number of words remaining in a circular buffer
;
; On entry, AR1 points to the buffer structure. On exit, the AC contains
; the number of words remaining in the buffer. ARP must point to AR1.
;
GETREM	LAC	*+		; compute upper - lower (buffer size)
	SUB	*+
	SACL	TMP
	LAC	*+		; compute get - put
	SUB	*+
	NEG
	BGZ	REM11		; branch if wrap
	ADD	TMP		; no wrap. add upper - lower
REM11	RET

;
; Routine to flush buffers
;
; This is used for initialization and switching between transmit and
; receive. It cleans up the buffers and conditions the modem for receive
; or transmit analog loopback. ARP must point to AR1.
;
FLUSH	LRLK	AR1,INPBCB	; preset input buffer structure
	LALK	INPBUF+OUTSIZ
	SACL	*+
	LALK	INPBUF
	SACL	*+
	SACL	*+
	SACL	*+
	LRLK	AR1,OUTBCB	; preset output buffer structure
	LALK	OUTBUF+OUTSIZ
	SACL	*+
	LALK	OUTBUF
	SACL	*+
	SACL	*+
	SACL	*+
	ZAC
	SACL	STATUS		; kill status bits
	SACL	TUNE		; kill radio control bits
	SACL	AUTCTR		; reset autostart
	LALK	ALPHA<<7	; reset encoders
	SACL	XCHCTR
	SACL	KEY+1
	SACL	KEY+2
	RET

;
; Routine to convert Baudot character to ASCII character
;
; On entry, AC contains a 5-bit Baudot code. On exit, AC contains the 7-
; bit ASCII code or zero if no corresponding ASCII code exists.
;
BDT2ASC	SUBK	SP		; is char space
	BNZ	BDT82		; branch if no
	BIT	MODE,RUSMOD	; yes. is unshift-on-space set
	BBZ	BDT82		; branch if no
	LALK	~(1<<RSHSTA)	; yes. clear shift bit
	AND	STATUS
	SACL	STATUS
	ZAC
BDT82	SUBK	FIGS-SP		; is char FIGS
	BNZ	BDT83		; branch if no
	LALK	1<<RSHSTA	; yes. set shift bit
	OR	STATUS
	SACL	STATUS
	ZAC
	RET
;
BDT83	SUBK	LTRS-FIGS	; is char LTRS
	BNZ	BDT84		; branch if no
	LALK	~(1<<RSHSTA)	; yes. clear shift bit
	AND	STATUS
	SACL	STATUS
	ZAC
	RET
;
BDT84	ADDK	LTRS		; restore char
	ADLK	BAUDOT		; translate Baudot to ASCII
	TBLR	TMP
	BIT	STATUS,RSHSTA	; is this LTRS shift
	BBNZ	BDT85		; branch if no
	LAC	TMP,16-8	; yes. use MSB
	SACH	TMP
BDT85	LAC	TMP
	ANDK	7Fh		; mask off junque
	RET

;
; Routine to convert ASCII character to Baudot character
;
; On entry, AC contains a 7-bit ASCII code. On exit, the AC contains the
; 5-bit Baudot code and LTRS/FIGS shift code, if needed, as two eight-
; bit bytes in a 16-bit word. If the shift code is not needed, the
; Baudot character is in the least significant byte; otherwise, the
; shift code is in that byte and the Baudot code is in the most
; significant byte. Before translation, lowercase ASCII characters are
; converted to uppercase and the parity bit is stripped. Except for the
; following, all other undefined ASCII codes are translated as zero.
;
; EOT	LTRS	close down transmitter immediately
; ACK	LTRS	close down transmitter when buffer empty
; ENQ	WRU	trigger answerback at receiver
; SYN	LTRS	use for fill if necessary
;
ASC2BDT	ANDK	07Fh		; unparity char
	SUBK	060h		; uncase char
	BLZ	ASC11
	SUBK	020h
ASC11	ADDK	060h
	SACL	STS
	SFR			; translate to Baudot
	ADLK	ASCII
	TBLR	TMP
	BIT	STS,0		; is this MSB 
	BBNZ	ASC13		; branch if no
	LAC	TMP,16-8	; yes. shift MSB to LSB
	SACH	TMP
ASC13	LAC	TMP		; mask off MSB bits
	ANDK	01Fh
	SACL	STS
	BIT	STATUS,XSHSTA	; is print in FIGS shift
	BBZ	ASC15		; branch if no (LTRS shift)
	BIT	TMP,LCTST	; yes. does char need LTRS shift
	BBZ	ASC17		; branch if no
	SUBK	SP		; yes. is char space
	BNZ	ASC12		; branch if no
	LAC	STS		; yes. restore char
	BIT	MODE,XUSMOD	; is unshift-on-space set
	BBZ	ASC17		; branch if no
ASC12	LALK	~(1<<XSHSTA)	; yes. set LTRS shift
	AND	STATUS
	SACL	STATUS
	LAC	STS,8		; send LTRS before char
	ORK	LTRS
	B	ASC17
;
ASC15	BIT	TMP,UCTST	; LTRS shift. does char need FIGS shift
	BBZ	ASC17		; branch if no
	LALK	1<<XSHSTA	; yes. set FIGS shift
	OR	STATUS
	SACL	STATUS
	LAC	STS,8		; send FIGS before char
	ORK	FIGS
ASC17	RET

;
; Routine to convert CCIR 476 code to Baudot character
;
; The CCIR 476 code vector developed by the baseband processing routine
; is stored in program memory in the coefficient page, while the
; correlation coefficients are stored in data memory. This routine
; correlates the seven-bit original code vector vector-summed with the
; seven-bit repeat character (delayed five characters) with each of the
; 35 possible CCIR 476 codes and selects the one with the largest
; correlation function. The corresponding Baudot character is returned
; in the AC. The routine returns 00h-1Fh for Baudot, 20h-22h for CCIR
; codes with no Baudot equivalent. In addition, the symbol (character)
; quality is computed as the distance normalized by the RMS signal. Note
; that we have already correlated the sync sequence and calculated its
; distance (ERSDST).
;
; Variables
; TORDLD/P	received bit vector (six seven-bit characters)
; TORTAB	correlation coefficients for 35 seven-bit characters
; ERSDST	erasure distance
;
; In conventional (hard decision) decoding, the error performance of the
; CCIR 476 code can be analyzed as follows. All seven-bit CCIR
; characters have weight four, so that the 14-bit CCIR code has minimum
; distance four. An error-free CCIR codeword, consisting of two
; identical CCIR characters, has a correlation function of 14, while its
; bit complement has a value of -14. A single bit error in one of the
; two CCIR characters results in a maximum value of 12 and is
; correctable. A two-bit error results in a maximum value of 10 and is
; not neccessarily correctable. This could occur if each of the two
; characters had weight four, but were distance two from each other.
;
CCR2BDT	LARP	AR1		; initialize
	ZAC
	SACL	TMP
	LRLK	AR1,TORTAB	; initialize for correlation
	LRLK	AR0,35-1
CCR12	ZAC			; correlate bit vector and weight vector
	MPYK	0
	RPTK	7-1
	MAC	TORDLP,*+	; multiply and add repetition position
	SBRK	7
	RPTK	7-1
	MAC	TORDLP+(7*5),*+	; multiply and add original position
	APAC			; accumulate final product
	SUBH	ERSDST		; is it greatest
	BLZ	CCR13		; branch if no
	ADDH	ERSDST		; yes. remember things
	SACH	ERSDST
	SAR	AR0,TMP
CCR13	LARP	AR0
	BANZ	CCR12,*-,AR1	; repeat for 35 CCIR codewords
	CALL	SUMSQ		; calculate sum of squares
	LAC	ERSDST
	SACL	AUTSIG
	CALL	ASTART		; manufacture autostart signal
	LACK	35-1		; figure Baudot char
	SUB	TMP
	RET

;
; Routine to manufacture autostart signal
;
; This routine processes the distance and signal squares of the
; character, integrates them in a boxcar integrator, and constructs the
; autostart signal, which is the ratio of the average distance to the
; RMS signal. It also cleans up the erasure distance and autostart
; counters.
;
; Variables
; AUTSIG	autostart signal
; ASCOPE	autostart gate
; SQRACC	(2) sum of squares
; AUTITG	autostart signal integrator
; AUNITG	autostart noise integrator
; AUTDEL	autostart delay line
;
; Note that the autostart signal (AUTSIG), derived from the character
; distance, and the sum of signal squares (SQRACC) have already been
; determined. This routine scales these data and stuff them in the delay
; line (AUTDEL). Then it accumulates the totals and finds the square
; root of the squares. It then divides the signal accumulation by the
; RMS accumulation to construct the distance signal. Finally, it
; normalizes the erasure distance.
;
ASTART	LAR	AR1,AUTPTR	; get delay line pointer
	LALK	32768/AUTSIZ	; compute scale factor
	SACL	STMP1
	LT	STMP1
	MPY	AUTSIG
	PAC
	SACH	STMP2
	PSHD	STMP2
	ZALH	STMP2		; add signal
	SUBH	*
	POPD	*+
	ADDH	AUTITG 
	SACH	AUTITG
	LT	STMP1		; add signal squares
	MPY	SQRACC
	PAC
	SACH	STMP2
	PSHD	STMP2
	ZALH	STMP2
	SUBH	*
	POPD	*+
	ADDH	AUNITG 
	SACH	AUNITG
	LRLK	AR0,AUTDEL+(2*AUTSIZ) ; wrap delay line pointer
	CMPR
	BBZ	CCR20
	LRLK	AR1,AUTDEL
CCR20	SAR	AR1,AUTPTR	; update delay line pointer
	LRLK	AR1,U_ADDR+AUNITG ; find square root
	CALL	SQRT
	SACL	STMP1
	ZALH	AUTITG		; compute autostart distance
	RPTK	15-1
	SUBC	STMP1
	NEG			; set autostart gate
	ADD	ATHRESH
	SACL	ASCOPE
	LRLK	AR1,U_ADDR+SQRACC ; find square root
	CALL	SQRT
	SACL	STMP1
	ZALH	ERSDST		; compute erasure distance
	RPTK	15-1
	SUBC	STMP1
	SACL	ERSDST
	LAC	ETHRESH		; set erasure gate
	SUB	ERSDST
	SACL	ESCOPE
	LAC	AUTCTR		; downdate autostart counter
	BZ	CCR22
	SUBK	1
	SACL	AUTCTR
CCR22	RET

; Compute sum-of-squares of the received samples in the SITOR delay
; line. We use the RMS signal as a normalizing factor.
;
; Variables
; SQRACC	(2) sum of squares
; TORDLD	received bit vector (six seven-bit characters)
; ERSDST	max survivor distance
;
SUMSQ	LARP	AR1		; initialize
	ZAC
	SACH	SQRACC
	SACL	SQRACC+1
	LALK	17515		; 2*32768/sqrt(14)
	SACL	STMP2
	CNFD			; map delay line to data mem
	LRLK	AR0,7-1
	LRLK	AR1,TORDLD	; point to repeat position
CCR21	LT	*		; adjust for scaling
	MPY	STMP2
	PAC
	SACH	STMP1
	LT	STMP1		; square and add
	MPY	STMP1
	PAC
	ADDH	SQRACC
	ADD	SQRACC+1
	SACH	SQRACC
	SACL	SQRACC+1
	ADRK	7*5		; point to original position
	LT	*		; adjust for scaling
	MPY	STMP2
	PAC
	SACH	STMP1
	LT	STMP1		; square and add
	MPY	STMP1
	PAC
	ADDH	SQRACC
	ADD	SQRACC+1
	SACH	SQRACC
	SACL	SQRACC+1
	SBRK	(7*5)-1		; point to repeat position
	LARP	AR0		; are all samples procssed
	BANZ	CCR21,*-,AR1	; branch if no
	CNFP			; map delay line to program mem
	RET

;
; Routine to convert Baudot character to CCIR 476 character
;
; The Baudot character is in the AC on entry, the CCIR 476 character is
; in the AC on exit.
;
BDT2CCR	ANDK	1Fh		; determine table address
	ADLK	CCIR
	TBLR	TMP		; get CCIR character
	LAC	TMP
	RET

;
; Routine to construct CCIR 476 decoding coefficient table
;
; The table has 35 vectors of seven words each, in order as the Baudot
; character equivalent, plus three special characters. It is constructed
; from the CCIR table, where each zero bit corresponds to a negative
; coefficient and each one bit corresponds to a positive coefficient.
;
TINIT	LARP	AR1		; AR1 points to coefficient table
	LRLK	AR1,TORTAB+(35*7)-1
	LALK	CCIR+35-1
	SACL	STS
TIN13	LAC	STS		; get code vector
	TBLR	TMP
	SUBK	1
	SACL	STS
	LRLK	AR0,7-1		; loop over seven bits
TIN12	LAC	TMP		; shift code right
	ROR
	SACL	TMP
	LALK	32768/7		; save signed coefficient
	BC	TIN11
	NEG
TIN11	SACL	*-
	LARP	AR0
	BANZ	TIN12,*-,AR1	; continue for seven coefficients
	LAC	STS		; is this last code
	SBLK	CCIR
	BGEZ	TIN13		; branch if no	
	RET			; yes. done here

;
; Subroutine to control panel LEDs
;
; This runs only when nothing else is waiting for processing. It
; controls the status LEDs and radio control lines. It also watches the
; number of words remaining in the output buffer and, if it falls below
; the red limit (ZONE), it drops the UART RTS (CTS at the connector)
; line and blinks LED 7. If the error bit is set (ERRSTA), it blinks LED
; 2. The remaining logical nastiness is indescribable.
;
LIGHTS	LARP	AR1		; AR1 means business
	LAC	FLASH		; bump the flash
	ADDK	1
	SACL	FLASH
;
; Assemble TNC_OUTPUT bits and LEDs 5-7
;
	LRLK	AR1,OUTBCB	; get number of output words remaining
	CALL	GETREM
	LDPK	B_PAGE		; go where the buffer action is
	SBLK	ZONE 
	BLZ	PAN11		; branch if red
	SBLK	ZONE 
	BLZ	PAN12		; branch if yellow
	LALK	UART_MCR_RTS	; green. raise RTS (CTS at connector)
	OR	UARTC
	SACL	UARTC
	B	PAN12
;
PAN11	LALK	~UART_MCR_RTS	; red. drop RTS (CTS at connector)
	AND	UARTC
	SACL	UARTC
PAN12	LALK	UART_MCR_RTS	; is RTS up
	AND	UARTC
	BNZ	PAN12A		; branch if yes
	LALK	~(LED_205+LED_206+LED_207) ; no. LEDs 5-7 off
	AND	TNC_BUF
	LDPK	U_PAGE
	BIT	FLASH,11	; flash LED 7
	BBZ	PAN12B
	ORK	LED_207
	B	PAN12B
;
PAN12A	LALK	~(LED_205+LED_206+LED_207) ; LEDs 5-7 off
	AND	TNC_BUF
	LDPK	U_PAGE
	BIT	KEY,0		; is this space
	BBNZ	PAN12B		; branch if no
	ORK	LED_207		; yes. LED 7 on
PAN12B	SACL	UTMP2
	LAC	CSCOPE		; is carrier up
	BGEZ	PAN21		; branch if no
	LAC	LEDREF,4	; yes. is mark > 15/16 threshold
	SUB	LEDREF
	SUB	MARK,4
	BGZ	PAN13		; branch if no
	LALK	LED_205		; yes. LED 5 on
	OR	UTMP2
	SACL	UTMP2
PAN13	LAC	LEDREF,4	; is space > 15/16 threshold
	SUB	LEDREF
	SUB	SPACE,4
	BGZ	PAN21		; branch if no
	LALK	LED_206		; yes. LED 6 on
	OR	UTMP2
	SACL	UTMP2
;
; Assemble RADIO_CTRL port bits and LEDs 3-4
;
PAN21	LAC	TUNE		; select radio port
	BIT	MODE,PORMOD
	BBNZ	PAN22
	ANDK	P1_MSK		; port 1
	B	PAN23
;
PAN22	ANDK	P2_MSK		; port 2
PAN23	SACL	UTMP1
	LAC	CSCOPE		; is signal gate unblocked
	BGEZ	PAN27		; branch if no
	LALK	LED_203		; yes. LED 3 on
	OR	UTMP1
	SACL	UTMP1
	LAC	ESCOPE		; is erasure distance ok
	BLZ	PAN24		; branch if yes
	BIT	MODE,AUTMOD	; no. is autostart enabled
	BBZ	PAN27		; branch if no
	LAC	AUTCTR		; yes. is autostart dead
	BNZ	PAN27		; branch if no
	BIT	FLASH,12	; yes. flash LED 4
	BBZ	PAN27
	B	PAN25
;
PAN24	BIT	STATUS,SYNSTA	; is this sync mode
	BBNZ	PAN25		; branch if yes
	BIT	MODE,TORMOD	; no. is this SITOR mode
	BBNZ	PAN27		; branch if yes
	BIT	KEY,0		; no. is output mark
	BBNZ	PAN27		; branch if yes
PAN25	LALK	LED_204		; no. LED 4 on
	OR	UTMP1
	SACL	UTMP1
PAN27	LAC	UTMP1
	LDPK	B_PAGE
	SACL	PDC_BUF
	OUT     PDC_BUF,RADIO_CTRL ; LEDs 3-4
	LDPK	U_PAGE
	LAC	UTMP2
	LDPK	B_PAGE
	SACL	TNC_BUF
	OUT	TNC_BUF,TNC_OUTPUT ; LEDs 5-7
	LDPK	U_PAGE
	RET

;
; Routine to generate scope sync signals
;
; The ACCUM signal can be monitored on a DC-coupled scope connected to
; the AIO output using the d5 command. This is a very useful tuning
; indicator and handy for performance assessment. At the speaker output,
; the AIO output signal sounds something like a Teletype Model 12 when
; piped to a speaker.
;
MODEL12	LARP	AR1
	LAC	BITCTR		; is input in progress
	BZ	MOD11		; branch if no
	LAC	SRVCTR		; yes. is this a bit time
	SUBK	1
	BGZ	MOD10		; branch if no
	LAC	BITCTR		; yes. update bit counter
	SUBK	1
	SACL	BITCTR
	LALK	-LIMIT/2	; make bit strobe
	SACL	SSYNC
	LACK	BINSIZ		; update sample counter
MOD10	SACL	SRVCTR
MOD11	LRLK	AR1,SRVBGN	; point to youngest bit for display
	BIT	MODE,TORMOD	; is this SITOR
	BBNZ	MOD12		; branch if yes
	ADRK	7*BINSIZ	; no. point to oldest bit
MOD12	LAC	*		; generate tick-rumble
	ADD	SSYNC
	SACL	ACCUM
	ZAC			; reset for next time
	SACL	SSYNC
	RET

;
; Routine to turn selected LED on and select AIO input
;
LEDON	OR	PIO_BUF		; set LED bit
	B	LED10

;
; Routine to turn selected LED off and select AIO input
;
LEDOFF	CMPL			; clear LED bit
	AND	PIO_BUF
LED10	ANDK	0C0h		; clear mux bits
	BIT	STATUS,ERRSTA	; is error flag on
	BBZ	LED20		; branch if no
	BIT	FLASH,11	; yes. flash LED 2
	BBZ	LED20
	ANDK	~LED_202
LED20	BIT	STATUS,CTSSTA	; is this transmit mode
	BBZ	LED12		; branch if no
	BIT	MODE,FDXMOD	; yes. is this full duplex
	BBNZ	LED12		; branch if yes
	ORK	AIO_ENABLE+GAIN0 ; no select transmitter monitor
	BIT	MODE,PORMOD
	BBNZ	LED11
	ORK	XMIT_1		; port 1
	B	LED14
;
LED11	ORK	XMIT_2		; port 2
	B	LED14
;
LED12	ORK	AIO_ENABLE	; select receiver monitor
	OR	AGAIN
	BIT	MODE,PORMOD
	BBNZ	LED13
	ORK	REC_IN_1	; port 1
	B	LED14
;
LED13	ORK	REC_IN_2	; port 2
LED14	SACL	PIO_BUF
	OUT	PIO_BUF,RADIO_GAIN ; (U_PAGE must be 8 for hardware)
	RET

;
; Wait loops (used only at initialization)
;
WAIT4	LARK	AR2,0FFh
	CALL	WAIT2
	CALL	WAIT2
	RET

WAIT2	LARK	AR2,0FFh
	CALL	WAIT1
	CALL	WAIT1
	RET

WAIT1	LARK	AR2,0FFh
WAIT_A	LARK	AR3,0FFh
WAIT_B	LARP	AR3
	BANZ	WAIT_B		; loop through count
	LARP	AR2
	BANZ	WAIT_A		; loop through count
	RET

	.MSFIRST

;
; Command tables
;
ST00	.WORD	GETPSH,ST40,'A'	; adjust autostart threshold and gate
	.WORD	GETPSH,ST20,'B'	; select baud rate
	.WORD	GETPSH,ST41,'C'	; adjust carrier threshold and channel
	.WORD	GETPSH,ST10,'D'	; select display/output
	.WORD	GETPSH,ST60,'E' ; digital/analog echo
	.WORD	GETPSH,ST51,'G'	; adjust filter gain
	.WORD	CMDTOG,1<<REVMOD,'I' ; reverse shift (toggle)
	.WORD	GETPSH,ST30,'L'	; select limiter gain
	.WORD	GETPSH,ST70,'M'	; select mode, filter and shift
	.WORD	GETPSH,ST15,'O'	; select output shift
	.WORD	GETPSH,ST25,'P'	; select radio port
	.WORD	CMDTOG,1<<ERSMOD,'Q' ; suppress erasures (toggle)
	.WORD	RESET,0,'R'	; master reset
	.WORD	CMDTOG,1<<SYNMOD,'S' ; sync enable (toggle)
	.WORD	GETPSH,ST50,'T'	; tune radio
	.WORD	GETPSH,ST35,'U'	; select unshift-on-space
	.WORD	GETPSH,ST45,'V'	; adjust erasure threshold
	.WORD	CMDX,0,'X'	; begin transmission
	.WORD	CMDY,0,'Y'	; enable SITOR mode B (FEC)
	.WORD	GET12,0,0	; default
;
; Select baud rate b{+-1-6}
;
ST20	.WORD	CMDB,BAUD1,'1'	; 100 baud
	.WORD	CMDB,BAUD2,'2'	; 75 baud
	.WORD	CMDB,BAUD3,'3'	; 50 baud
	.WORD	CMDB,BAUD4,'4'	; 45.45 baud
	.WORD	CMDB,BAUD5,'5'	; 25 baud
	.WORD	CMDB,BAUD6,'6'	; 10 baud
	.WORD	CMDBP,0,'+'	; increase
	.WORD	CMDBP,0,'='	; increase (LC)
	.WORD	CMDBN,0,'-'	; decrease
	.WORD	CMDBN,0,'_'	; decrease (UC)
	.WORD	GETPOP,0,0	; default
;
; Select mode, filter and shift m{+-1-8}
;
ST70	.WORD	CMDM,SHIFT1,'1'	; IIR filter, 850 Hz shift
	.WORD	CMDM,SHIFT2,'2'	; IIR filter, 600 Hz shift
	.WORD	CMDM,SHIFT3,'3'	; IIR filter, 400 Hz shift
	.WORD	CMDM,SHIFT4,'4'	; IIR filter, 170 Hz shift
	.WORD	CMDM,SHIFT5,'5'	; MF filter, 850 Hz shift
	.WORD	CMDM,SHIFT6,'6'	; MF filter, 600 Hz shift
	.WORD	CMDM,SHIFT7,'7'	; MF filter, 400 Hz shift
	.WORD	CMDM,SHIFT8,'8'	; MF filter, 170 Hz shift
	.WORD	CMDSP,0,'+'	; increase
	.WORD	CMDSP,0,'='	; increase (LC)
	.WORD	CMDSN,0,'-'	; decrease
	.WORD	CMDSN,0,'_'	; decrease (UC)
	.WORD	GETPOP,0,0	; default
;
; Select limiter gain l{0-5}
;
ST30	.WORD	CMDL,0,'0'	; 0dB
	.WORD	CMDL,1,'1'	; 6dB
	.WORD	CMDL,2,'2'	; 12dB
	.WORD	CMDL,3,'3'	; 18dB
	.WORD	CMDL,4,'4'	; 24dB
	.WORD	CMDL,5,'5'	; 30dB
	.WORD	GETPOP,0,0	; defaultt
;
; Select display (output) d<0-9>
;
ST10	.WORD	CMDD,U_ADDR+STS,'0' ; AFSK generator
	.WORD	CMDD,U_ADDR+INPSIG,'1' ; AIO input signal
	.WORD	CMDD,U_ADDR+FLTSIG,'2' ; filtered/limited input signal
	.WORD	CMDD,U_ADDR+MARK,'3' ; mark signal
	.WORD	CMDD,U_ADDR+SPACE,'4' ; space signal
	.WORD	CMDD,U_ADDR+ACCUM,'5' ; detector output
	.WORD	CMDD,U_ADDR+CSCOPE,'6' ; carrier gate
	.WORD	CMDD,U_ADDR+ESCOPE,'7' ; erasure gate
	.WORD	CMDD,U_ADDR+ASCOPE,'8' ; autostart gate
	.WORD	CMDD,U_ADDR+XPHASE,'9' ; VCO phase signa distance
	.WORD	CMDD,U_ADDR+PEEK1,'-' ; peekaboo 1 (LC)
	.WORD	CMDD,U_ADDR+PEEK2,'_' ; peekaboo 2 (UC)
	.WORD	GETPOP,0,0	; default
;
; Select analog/digital loopback echo e{1-3}
;
ST60	.WORD	CMDE,0,'1'	; analog loopback echo
	.WORD	CMDE,1<<ECHMOD,'2' ; digital loopback echo
	.WORD	CMDE,1<<FDXMOD,'3' ; full duplex
	.WORD	GETPOP,0,0	; default
;
; Select radio port {p1-2}
;
ST25	.WORD	CMDP1,0,'1'	; radio port 1
	.WORD	CMDP2,0,'2'	; radio port 2
	.WORD	GETPOP,0,0	; default
;
; Select output shift {o0-2}
;
ST15	.WORD	CMDO0,0,'0'	; follow input shift
	.WORD	CMDO1,0,'1'	; 170-Hz shift
	.WORD	CMDO2,0,'2'	; 850-Hz shift
	.WORD	GETPOP,0,0	; default
;
; Select unshift-on-space {u0-2}
;
ST35	.WORD	CMDU,0,'0'	; unshift off
	.WORD	CMDU,1<<RUSMOD,'1' ; receive unshift
	.WORD	CMDU,1<<XUSMOD,'2' ; transmit unshift
	.WORD	CMDU,(1<<RUSMOD)+(1<<XUSMOD),'3' ; transmit/receive unshift
	.WORD	GETPOP,0,0	; default
;
; Tune radio t{+-}
;
ST50	.WORD	CMDTP,0,'+'	; increase
	.WORD	CMDTP,0,'='	; increase (LC)
	.WORD	CMDTN,0,'-'	; decrease
	.WORD	CMDTN,0,'_'	; decrease (UC)
	.WORD	GETPOP,0,0	; default
;
; Adjust gain g{+-,0-7}
;
ST51	.WORD	CMDGG,GAIN0,'0' ; unity gain
	.WORD	CMDGG,GAIN1,'1' ; 6dB gain
	.WORD	CMDGG,GAIN2,'2' ; 12dB gain
	.WORD	CMDGG,GAIN3,'3' ; 18dB gain
	.WORD	CMDGG,GAIN4,'4' ; 24dB gain
	.WORD	CMDGG,GAIN5,'5' ; 30dB gain
	.WORD	CMDGG,GAIN6,'6' ; 36dB gain
	.WORD	CMDGG,GAIN7,'7' ; comparator
	.WORD	CMDGN,0,'+'	; increase
	.WORD	CMDGN,0,'='	; increase (LC)
	.WORD	CMDGP,0,'-'	; decrease
	.WORD	CMDGP,0,'_'	; decrease (UC)
	.WORD	GETPOP,0,0	; default
;
; Adjust autostart threshold a{+-,0-1}
;
ST40	.WORD	CMDAN,0,'+'	; increase
	.WORD	CMDAN,0,'='	; increase (LC)
	.WORD	CMDAP,0,'-'	; decrease
	.WORD	CMDAP,0,'_'	; decrease (UC)
	.WORD	CMDA,0,'0'	; autostart off
	.WORD	CMDA,1<<AUTMOD,'1' ; autostart gate
	.WORD	GETPOP,0,0	; default
;
; Adjust erasure threshold v{+-}
;
ST45	.WORD	CMDVN,0,'+'	; increase
	.WORD	CMDVN,0,'='	; increase (LC)
	.WORD	CMDVP,0,'-'	; decrease
	.WORD	CMDVP,0,'_'	; decrease (UC)
	.WORD	GETPOP,0,0	; default
;
; Adjust carrier threshold c{+-012}
;
ST41	.WORD	CMDCN,0,'+'	; increase
	.WORD	CMDCN,0,'='	; increase (LC)
	.WORD	CMDCP,0,'-'	; decrease
	.WORD	CMDCP,0,'_'	; decrease (UC)
	.WORD	CMDC,0,'0'	; mark/space
	.WORD	CMDC,1<<SPCMOD,'1' ; mark only (space disable)
	.WORD	CMDC,1<<MRKMOD,'2' ; space only (mark disable)
	.WORD	GETPOP,0,0	; default
;
; Initialization table. This is used to configure the AIO at initial
; startup.
;
INITAB	.WORD	3		; 0 load AIO configuration
	.WORD	CFGC		; 1
	.WORD	3		; 2 load TB register
	.WORD	CFGB		; 3
	.WORD	3		; 4 load TA register
	.WORD	CFGA		; 5
	.WORD	0		; 6 terminator

;
; Baud rate data tables. Note that 74.2 baud is rounded to 75 baud,
; since almost all RTTY today uses electronic UARTs. The first word is
; the number of AIO samples per bit sample, where there are 8 samples
; per bit. The next two words establish the matched filter size and
; gain, respectively. Note that the FAX rates are synchronous with
; submultiples of the AIO sample frequency to minimize aliasing.
;
; 100 baud (132 WPM)
;
BAUD1	.WORD	1250		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	79*4		; MF size * 4 (SFREQ / r)
	.WORD	830		; MF gain (2^16 / size)
;
; 75 Baud (100 WPM)
;
BAUD2	.WORD	1667		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	106*4		; MF size * 4 (SFREQ / r)
	.WORD	618		; MF gain (2^16 / size)
;
; 50 baud (66 WPM)
;
BAUD3	.WORD	2500		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	159*4		; MF size * 4 (SFREQ / r)
	.WORD	412		; MF gain (2^16 / size)
;
; 45.45 baud (60 WPM)
;
BAUD4	.WORD	2750		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	175*4		; MF size * 4 (SFREQ / r)
	.WORD	374		; MF gain (2^16 / size)
;
; 25 baud (33 WPM)
;
BAUD5	.WORD	5000		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	318*4		; MF size * 4 (SFREQ / r)
	.WORD	206		; MF gain (2^16 / size)
;
; 10 baud (13.2 WPM)
;
BAUD6	.WORD	12500		; AIO samples (1000000 / (r * BINSIZ))
	.WORD	745*4		; MF size * 4 (SFREQ / r)
	.WORD	88		; MF gain (2^16 / size)

;
; Tone frequency data tables. These are used to initialize variables for
; various combinations of mode, filter and spacing. The first two words
; control the input mark and space carrier oscillators, respectively.
; The third word controls various mode bits. The fourth word sets the
; filter gain n+m, where n is the calculated gain factor and m is a trim
; value. The fifth and sixth words control the output AFSK mark and
; space carrier oscillators. The MF filter gain is set empirically at
; 2295 Hz with the limiter switched off so that the signal LED (LED 4)
; comes on with the mark LED (LED 5). The FIR filter gains are set for
; 800mV peak mark signal at the input clipper threshold; the IIR filter
; gains are set so the mark signal is the same as the MF value at each
; shift.
;
; Note the narrrow-shift IIR tones are centered 125 Hz wide of the
; working shift, in order to improve linearity; e.g., the mark filter is
; centered on 2420 Hz, the space filter on 2000 Hz. For all shifts, the
; center frequency of the discriminator is half way between the upper
; and lower 3-dB points of the corresponding FIR bandpass filter. This
; bit of nonsense matches the AEA PK-232 TNC.
;
; IIR filter, center frequency 1700 Hz, wide shift 850 Hz
;
;
SHIFT1	.WORD	IFILT		; mark filter
	.WORD	-(1275*FAC)	; mark in
	.WORD	IFILT		; space filter
	.WORD	-(2125*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	34		; filter gain
;
; IIR filter, center frequency 1700 Hz, mid shift 600 Hz
;
SHIFT2	.WORD	IFILT		; mark filter
	.WORD	-(1400*FAC)	; mark in
	.WORD	IFILT		; space filter
	.WORD	-(2000*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	34		; filter gain

;
; IIR filter, center frequency 1700 Hz, mid shift 400 Hz
;
SHIFT3	.WORD	IFILT		; mark filter
	.WORD	-(1500*FAC)	; mark in
	.WORD	IFILT		; space filter
	.WORD	-(1900*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	37		; filter gain
;
; IIR filter, center frequency 2210 Hz, narrow shift 170 Hz
;
SHIFT4	.WORD	IFILT		; mark filter
	.WORD	-(2000*FAC)	; mark in
	.WORD	IFILT		; space filter
	.WORD	-(2420*FAC)	; space in
	.WORD	1<<NARMOD	; narrow filter
	.WORD	42		; filter gain
;
; MF filter, center frequency 1700 Hz, wide shift 850 Hz
;
SHIFT5	.WORD	MFILT		; mark filter
	.WORD	-(1275*FAC)	; mark in
	.WORD	MFILT		; space filter
	.WORD	-(2125*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	256+8		; filter gain
;
; MF filter, center frequency 1700 Hz, mid shift 600 Hz
;
SHIFT6	.WORD	MFILT		; mark filter
	.WORD	-(1400*FAC)	; mark in
	.WORD	MFILT		; space filter
	.WORD	-(2000*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	256+8		; filter gain
;
; MF filter, center frequency 1700 Hz, mid shift 400 Hz
;
SHIFT7	.WORD	MFILT		; mark filter
	.WORD	-(1500*FAC)	; mark in
	.WORD	MFILT		; space filter
	.WORD	-(1900*FAC)	; space in
	.WORD	0		; wide filter
	.WORD	256+11		; filter gain
;
; MF filter, center frequency 2210 Hz, narrow shift 170 Hz
;
SHIFT8	.WORD	MFILT		; mark filter
	.WORD	-(2125*FAC)	; mark in
	.WORD	MFILT		; space filter
	.WORD	-(2295*FAC)	; space in
	.WORD	1<<NARMOD	; narrow filter
	.WORD	256+18		; filter gain

;
; ITA-2 to ASCII translate table. LSB is FIGS shift, MSB is LTRS shift.
; Adapted from ARRL 1995 Handbook Table 30.41. Each line defines one
; word ".BYTE x,y", where x is the MSB and y is the LSB. The LTRS-shift
; Baudot characters are in the MSB, while the FIGS-shift ones are in the
; LSB.
;				     ALL  ITA2 US
;				     LTRS FIGS FIGS
BAUDOT	.BYTE	NUL,NUL		; 00 NUL  NUL  NUL
	.BYTE	'E','3'		; 01 E    3    3
	.BYTE	LF,LF		; 02 LF   LF   LF
	.BYTE	'A','-'		; 03 A    -    -
	.BYTE	' ',' '		; 04 SP   SP   SP
	.BYTE	'S',23h		; 05 S    '    BELL
	.BYTE	'I','8'		; 06 I    8    8
	.BYTE	'U','7'		; 07 U    7    7
	.BYTE	CR,CR		; 08 CR   CR   CR
	.BYTE	'D',ENQ		; 09 D    WRU  $
	.BYTE	'R','4'		; 0A R    4    4
	.BYTE	'J',BEL		; 0B J    BELL '
	.BYTE	'N',','		; 0C N    ,    ,
	.BYTE	'F','!'		; 0D F    NA   !
	.BYTE	'C',':'		; 0E C    :    :
	.BYTE	'K','('		; 0F K    (    (
	.BYTE	'T','5'		; 10 T    5    5
	.BYTE	'Z','+'		; 11 Z    +    "
	.BYTE	'L',')'		; 12 L    )    )
	.BYTE	'W','2'		; 13 W    2    2
	.BYTE	'H','#'		; 14 H    NA   #
	.BYTE	'Y','6'		; 15 Y    6    6
	.BYTE	'P','0'		; 16 P    0    0
	.BYTE	'Q','1'		; 17 Q    1    1
	.BYTE	'O','9'		; 18 O    9    9
	.BYTE	'B','?'		; 19 B    ?    ?
	.BYTE	'G','&'		; 1A G    NA   &
	.BYTE	NUL,NUL		; 1B FIGS FIGS FIGS
	.BYTE	'M','.'		; 1C M    .    .
	.BYTE	'X','/'		; 1D X    /    /
	.BYTE	'V','='		; 1E V    =    ;
	.BYTE	NUL,NUL		; 1F LTRS LTRS LTRS

;
; ASCII to ITA-2 translate table. Adapted from ARRL 1995 Handbook Table
; 30.44. Each line defines one word ".BYTE x,y", where x is the MSB and
; y is the LSB. The even-numbered ASCII characters are in the MSB, while
; the odd-numbered ones in the LSB. Note that the transmitter control
; characters EOT, ACK and SYN translated as Baudot LTRS. Note also that
; the format effectors CR and LF are transmitted in LTRS shift, and that
; SP is transmitted in LTRS shift if the print-erasure is enabed, in
; order to beautify copy under noisy conditions.
;
ASCII	.BYTE	0,0		; 00 NUL 01 SOH
	.BYTE	0,0		; 02 STX 03 ETX
	.BYTE	1Fh,09h+UC	; 04 EOT 05 ENQ (ENQ = Baudot WRU)
	.BYTE	1Fh,0Bh+UC	; 06 ACK 07 BEL
	.BYTE	0,0		; 08 BS  09 HT
	.BYTE	02h+LC,0	; 0A LF  0B VT
	.BYTE	0,08h+LC	; 0C FF  0D CR
	.BYTE	0,0		; 0E SO  0F SI
	.BYTE	0,0		; 10 DLE 11 DC1
	.BYTE	0,0		; 12 DC2 13 DC3
	.BYTE	0,0		; 14 DC4 15 NAK
	.BYTE	1Fh,0		; 16 SYN 17 ETB (SYN = Baudot LTRS)
	.BYTE	0,0		; 18 CAN 19 EM
	.BYTE	0,0		; 1A SUB 1B ESC
	.BYTE	0,0		; 1C FS  1D GS
	.BYTE	0,0		; 1E RS  1F US
 	.BYTE	04h+LC,0Dh+UC	; 20 SP  21 !
	.BYTE	0,14h+UC	; 22 "   23 #
	.BYTE	0,0		; 24 $   25 %
	.BYTE	1A+UC,05h+UC	; 26 &   27 '
	.BYTE	0Fh+UC,12h+UC	; 28 (   29 )
	.BYTE	0,11h+UC	; 2A *   2B +
	.BYTE	0Ch+UC,03H+UC	; 2C ,   2D -
	.BYTE	1Ch+UC,1Dh+UC	; 2E .   2F /
	.BYTE	16h+UC,17h+UC	; 30 0   31 1
	.BYTE	13h+UC,01h+UC	; 32 2   33 3
	.BYTE	0Ah+UC,10h+UC	; 34 4   35 5
	.BYTE	15h+UC,07h+UC	; 36 6   37 7
	.BYTE	06h+UC,18h+UC	; 38 8   39 9
	.BYTE	0Eh+UC,0	; 3A :   3B ;
	.BYTE	0,1Eh+UC	; 3C <   3D =
	.BYTE	0,19h+UC	; 3E >   3F ?
	.BYTE	0,03h+LC	; 40 @   41 A
	.BYTE	19h+LC,0Eh+LC	; 42 B   43 C
	.BYTE	09h+LC,01h+LC	; 44 D   45 E
	.BYTE	0Dh+LC,1Ah+LC	; 46 F   47 G
	.BYTE	14h+LC,06h+LC	; 48 H   49 I
	.BYTE	0Bh+LC,0Fh+LC	; 4A J   4B K
	.BYTE	12h+LC,1Ch+LC	; 4C L   4D M
	.BYTE	0Ch+LC,18h+LC	; 4E N   4F O
	.BYTE	16h+LC,17h+LC	; 50 P   51 Q
	.BYTE	0Ah+LC,05h+LC	; 52 R   53 S
	.BYTE	10h+LC,07h+LC	; 54 T   55 U
	.BYTE	1Eh+LC,13h+LC	; 56 V   57 W
	.BYTE	1Dh+LC,15h+LC	; 58 X   59 Y
	.BYTE	11h+LC,0	; 5A Z   5B [
	.BYTE	0,0		; 5C \   5D ]
	.BYTE	0,0		; 5E `   5F _

;
; ITA-2 to CCIR 476 translation table. Adapted from ARRL 1995
; Handbook Table 30.41 and Table 12.6. The CCIR 476 to ITA-2 translation
; table is built on-the-fly from this table.
;
CCIR	.WORD	1101010b	; 00 NUL (CS2)
	.WORD	1010110b	; 01 E
	.WORD	1101100b	; 02 LF
	.WORD	1000111b	; 03 A
	.WORD	1011100b	; 04 SP
	.WORD	1001011b	; 05 S
	.WORD	1001101b	; 06 I
	.WORD	1001110b	; 07 U
	.WORD	1111000b	; 08 CR
	.WORD	1010011b	; 09 D
	.WORD	1010101b	; 0A R
	.WORD	0010111b	; 0B J
	.WORD	1011001b	; 0C N (CS3)
	.WORD	0011011b	; 0D F
	.WORD	0011101b	; 0E C
	.WORD	0011110b	; 0F K
	.WORD	1110100b	; 10 T
	.WORD	1100011b	; 11 Z
	.WORD	1100101b	; 12 L (CS1)
	.WORD	0100111b	; 13 W
	.WORD	1101001b	; 14 H (CS5)
	.WORD	0101011b	; 15 Y
	.WORD	0101101b	; 16 P
	.WORD	0101110b	; 17 Q
	.WORD	1110001b	; 18 O
	.WORD	1110010b	; 19 B
	.WORD	0110101b	; 1A G (CS4)
	.WORD	0110110b	; 1B FIGS
	.WORD	0111001b	; 1C M
	.WORD	0111010b	; 1D X
	.WORD	0111100b	; 1E V
	.WORD	1011010b	; 1F LTRS
CCTRQ	.WORD	1100110b	; 20 signal repetition (RQ)
CCTALF	.WORD	0001111b	; 21 signal alpha (ALPHA)
CCTBET	.WORD	0110011b	; 22 signal beta

;
; The following filter coefficient tables were generated using Math
; Works Inc. Matlab Signal Processing Toolkit. The performance data were
; obtained directly from the rounded and truncated coefficients, in
; order to accurately model the roundoff error.
;
; IIR coefficients for 170-Hz shift LPF - Butterworth, q10/20 format,
; sample freq 8000, order 2, transition 125 Hz, 3dB 150 Hz, 6dB 188 Hz
;
IIR1	.WORD	-1734		; a2 q10
	.WORD	3755		; a1 q10
	.WORD	2048		; a0 q10
	.WORD	6711		; b0 q20
	.WORD	13421		; b1 q20
	.WORD	6711		; b2 q20

;
; The following two tables define the FIR coefficients for a 170-Hz
; bandpass filter and a 850-Hz bandpass filter. Both were produced using
; the Remez method, sample frequency 8000 Hz, order 70, q15 format.
;
; FIR coefficients for 170-Hz shift - design passband 2125-2295 Hz,
; transition band 300 Hz, 3dB 2031-2391 Hz, 6dB 1984-2438 Hz, stopbands
; 0-1828 and 2625-4000 Hz, max stopband ripple -50dB.
;
BPF170	.WORD	   -11		;   0
	.WORD	   -19		;   1
	.WORD	   -25		;   2
	.WORD	   -49		;   3
	.WORD	    97		;   4
	.WORD	    28		;   5
	.WORD	  -167		;   6
	.WORD	    19		;   7
	.WORD	   243		;   8
	.WORD	  -119		;   9
	.WORD	  -267		;  10
	.WORD	   237		;  11
	.WORD	   216		;  12
	.WORD	  -310		;  13
	.WORD	  -102		;  14
	.WORD	   270		;  15
	.WORD	    -2		;  16
	.WORD	   -88		;  17
	.WORD	   -15		;  18
	.WORD	  -189		;  19
	.WORD	   256		;  20
	.WORD	   432		;  21
	.WORD	  -752		;  22
	.WORD	  -465		;  23
	.WORD	  1420		;  24
	.WORD	   138		;  25
	.WORD	 -2064		;  26
	.WORD	   591		;  27
	.WORD	  2438		;  28
	.WORD	 -1613		;  29
	.WORD	 -2336		;  30
	.WORD	  2681		;  31
	.WORD	  1692		;  32
	.WORD	 -3491		;  33
	.WORD	  -618		;  34
	.WORD	  3792		;  35
	.WORD	  -618		;  36
	.WORD	 -3491		;  37
	.WORD	  1692		;  38
	.WORD	  2681		;  39
	.WORD	 -2336		;  40
	.WORD	 -1613		;  41
	.WORD	  2438		;  42
	.WORD	   591		;  43
	.WORD	 -2064		;  44
	.WORD	   138		;  45
	.WORD	  1420		;  46
	.WORD	  -465		;  47
	.WORD	  -752		;  48
	.WORD	   432		;  49
	.WORD	   256		;  50
	.WORD	  -189		;  51
	.WORD	   -15		;  52
	.WORD	   -88		;  53
	.WORD	    -2		;  54
	.WORD	   270		;  55
	.WORD	  -102		;  56
	.WORD	  -310		;  57
	.WORD	   216		;  58
	.WORD	   237		;  59
	.WORD	  -267		;  60
	.WORD	  -119		;  61
	.WORD	   243		;  62
	.WORD	    19		;  63
	.WORD	  -167		;  64
	.WORD	    28		;  65
	.WORD	    97		;  66
	.WORD	   -49		;  67
	.WORD	   -25		;  68
	.WORD	   -19		;  69
	.WORD	   -11		;  70

;
; FIR coefficients for 170-Hz shift - design passband 1275-2125 Hz,
; transition band 300 Hz, 3dB 1172-2234 Hz, 6dB 1141-2266 Hz, stopbands
; 0-984 and 2422-4000 Hz, max stopband ripple -48dB.
;
BPF850	.WORD	    24		;   0
	.WORD	   -14		;   1
	.WORD	    79		;   2
	.WORD	    27		;   3
	.WORD	  -128		;   4
	.WORD	  -104		;   5
	.WORD	    42		;   6
	.WORD	    11		;   7
	.WORD	     1		;   8
	.WORD	   197		;   9
	.WORD	   113		;  10
	.WORD	  -292		;  11
	.WORD	  -259		;  12
	.WORD	    99		;  13
	.WORD	    29		;  14
	.WORD	    -3		;  15
	.WORD	   467		;  16
	.WORD	   316		;  17
	.WORD	  -606		;  18
	.WORD	  -594		;  19
	.WORD	   181		;  20
	.WORD	    45		;  21
	.WORD	   -41		;  22
	.WORD	  1024		;  23
	.WORD	   810		;  24
	.WORD	 -1257		;  25
	.WORD	 -1410		;  26
	.WORD	   334		;  27
	.WORD	    57		;  28
	.WORD	  -218		;  29
	.WORD	  2876		;  30
	.WORD	  2917		;  31
	.WORD	 -4355		;  32
	.WORD	 -7261		;  33
	.WORD	  2122		;  34
	.WORD	  9423		;  35
	.WORD	  2122		;  36
	.WORD	 -7261		;  37
	.WORD	 -4355		;  38
	.WORD	  2917		;  39
	.WORD	  2876		;  40
	.WORD	  -218		;  41
	.WORD	    57		;  42
	.WORD	   334		;  43
	.WORD	 -1410		;  44
	.WORD	 -1257		;  45
	.WORD	   810		;  46
	.WORD	  1024		;  47
	.WORD	   -41		;  48
	.WORD	    45		;  49
	.WORD	   181		;  50
	.WORD	  -594		;  51
	.WORD	  -606		;  52
	.WORD	   316		;  53
	.WORD	   467		;  54
	.WORD	    -3		;  55
	.WORD	    29		;  56
	.WORD	    99		;  57
	.WORD	  -259		;  58
	.WORD	  -292		;  59
	.WORD	   113		;  60
	.WORD	   197		;  61
	.WORD	     1		;  62
	.WORD	    11		;  63
	.WORD	    42		;  64
	.WORD	  -104		;  65
	.WORD	  -128		;  66
	.WORD	    27		;  67
	.WORD	    79		;  68
	.WORD	   -14		;  69
	.WORD	    24		;  70

;
; Sine table in 3-degree increments (450 degrees), q15 format
;
SINTAB	.WORD	     0		;   0
	.WORD	  1715		;   3
	.WORD	  3425		;   6
	.WORD	  5126		;   9
	.WORD	  6813		;  12
	.WORD	  8481		;  15
	.WORD	 10126		;  18
	.WORD	 11743		;  21
	.WORD	 13328		;  24
	.WORD	 14876		;  27
	.WORD	 16384		;  30
	.WORD	 17847		;  33
	.WORD	 19261		;  36
	.WORD	 20622		;  39
	.WORD	 21926		;  42
	.WORD	 23170		;  45
	.WORD	 24351		;  48
	.WORD	 25466		;  51
	.WORD	 26510		;  54
	.WORD	 27482		;  57
	.WORD	 28378		;  60
	.WORD	 29197		;  63
	.WORD	 29935		;  66
	.WORD	 30592		;  69
	.WORD	 31164		;  72
	.WORD	 31651		;  75
	.WORD	 32052		;  78
	.WORD	 32365		;  81
	.WORD	 32588		;  84
	.WORD	 32723		;  87
	.WORD	 32767		;  90
	.WORD	 32723		;  93
	.WORD	 32588		;  96
	.WORD	 32365		;  99
	.WORD	 32052		; 102
	.WORD	 31651		; 105
	.WORD	 31164		; 108
	.WORD	 30592		; 111
	.WORD	 29935		; 114
	.WORD	 29197		; 117
	.WORD	 28378		; 120
	.WORD	 27482		; 123
	.WORD	 26510		; 126
	.WORD	 25466		; 129
	.WORD	 24351		; 132
	.WORD	 23170		; 135
	.WORD	 21926		; 138
	.WORD	 20622		; 141
	.WORD	 19261		; 144
	.WORD	 17847		; 147
	.WORD	 16384		; 150
	.WORD	 14876		; 153
	.WORD	 13328		; 156
	.WORD	 11743		; 159
	.WORD	 10126		; 162
	.WORD	  8481		; 165
	.WORD	  6813		; 168
	.WORD	  5126		; 171
	.WORD	  3425		; 174
	.WORD	  1715		; 177
	.WORD	     0		; 180
	.WORD	 -1715		; 183
	.WORD	 -3425		; 186
	.WORD	 -5126		; 189
	.WORD	 -6813		; 192
	.WORD	 -8481		; 195
	.WORD	-10126		; 198
	.WORD	-11743		; 201
	.WORD	-13328		; 204
	.WORD	-14876		; 207
	.WORD	-16384		; 210
	.WORD	-17847		; 213
	.WORD	-19261		; 216
	.WORD	-20622		; 219
	.WORD	-21926		; 222
	.WORD	-23170		; 225
	.WORD	-24351		; 228
	.WORD	-25466		; 231
	.WORD	-26510		; 234
	.WORD	-27482		; 237
	.WORD	-28378		; 240
	.WORD	-29197		; 243
	.WORD	-29935		; 246
	.WORD	-30592		; 249
	.WORD	-31164		; 252
	.WORD	-31651		; 255
	.WORD	-32052		; 258
	.WORD	-32365		; 261
	.WORD	-32588		; 264
	.WORD	-32723		; 267
	.WORD	-32768		; 270
	.WORD	-32723		; 273
	.WORD	-32588		; 276
	.WORD	-32365		; 279
	.WORD	-32052		; 282
	.WORD	-31651		; 285
	.WORD	-31164		; 288
	.WORD	-30592		; 291
	.WORD	-29935		; 294
	.WORD	-29197		; 297
	.WORD	-28378		; 300
	.WORD	-27482		; 303
	.WORD	-26510		; 306
	.WORD	-25466		; 309
	.WORD	-24351		; 312
	.WORD	-23170		; 315
	.WORD	-21926		; 318
	.WORD	-20622		; 321
	.WORD	-19261		; 324
	.WORD	-17847		; 327
	.WORD	-16384		; 330
	.WORD	-14876		; 333
	.WORD	-13328		; 336
	.WORD	-11743		; 339
	.WORD	-10126		; 342
	.WORD	 -8481		; 345
	.WORD	 -6813		; 348
	.WORD	 -5126		; 351
	.WORD	 -3425		; 354
	.WORD	 -1715		; 357
	.WORD	    -0		; 360
	.WORD	  1715		; 363
	.WORD	  3425		; 366
	.WORD	  5126		; 369
	.WORD	  6813		; 372
	.WORD	  8481		; 375
	.WORD	 10126		; 378
	.WORD	 11743		; 381
	.WORD	 13328		; 384
	.WORD	 14876		; 387
	.WORD	 16384		; 390
	.WORD	 17847		; 393
	.WORD	 19261		; 396
	.WORD	 20622		; 399
	.WORD	 21926		; 402
	.WORD	 23170		; 405
	.WORD	 24351		; 408
	.WORD	 25466		; 411
	.WORD	 26510		; 414
	.WORD	 27482		; 417
	.WORD	 28378		; 420
	.WORD	 29197		; 423
	.WORD	 29935		; 426
	.WORD	 30592		; 429
	.WORD	 31164		; 430
	.WORD	 31651		; 435
	.WORD	 32052		; 438
	.WORD	 32365		; 441
	.WORD	 32588		; 444
	.WORD	 32723		; 447
	.WORD	 32767		; 450
;
	.END			; end of program
